<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Page;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;

class PageController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Page::query()->with(['author', 'updatedByUser']);

        // Search
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                    ->orWhere('slug', 'like', "%{$search}%")
                    ->orWhere('content', 'like', "%{$search}%");
            });
        }

        // Filter by status
        if ($request->has('status') && $request->status != '') {
            $query->where('status', $request->status);
        }

        // Filter by featured
        if ($request->has('is_featured') && $request->is_featured != '') {
            $query->where('is_featured', $request->is_featured);
        }

        $pages = $query->orderBy('sort_order')->orderBy('created_at', 'desc')->paginate(15);

        return view('admin.pages.index', compact('pages'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.pages.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:pages,slug',
            'content' => 'required|string',
            'excerpt' => 'nullable|string',
            'status' => 'required|in:draft,published,private',
            'sort_order' => 'nullable|integer',
            'is_featured' => 'nullable|boolean',
            'published_at' => 'nullable|date',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
            'meta_keywords' => 'nullable|string',
            'meta_robots' => 'nullable|string|max:255',
            'canonical_url' => 'nullable|string|max:255',
            'schema_markup' => 'nullable|string',
            'featured_image' => 'nullable|string|max:255',
            'featured_image_file' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'featured_image_alt' => 'nullable|string|max:255',
            'featured_image_caption' => 'nullable|string|max:255',
            'banner_image' => 'nullable|string|max:255',
            'banner_image_file' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'banner_image_alt' => 'nullable|string|max:255',
        ]);

        // Generate slug if not provided
        if (empty($validated['slug'])) {
            $validated['slug'] = Str::slug($validated['title']);
        }

        // Handle featured image upload
        if ($request->hasFile('featured_image_file')) {
            $featuredImage = $request->file('featured_image_file');
            $featuredImageName = time() . '_featured_' . $featuredImage->getClientOriginalName();
            $featuredImage->move(public_path('uploads/pages'), $featuredImageName);
            $validated['featured_image'] = '/uploads/pages/' . $featuredImageName;
        }

        // Handle banner image upload
        if ($request->hasFile('banner_image_file')) {
            $bannerImage = $request->file('banner_image_file');
            $bannerImageName = time() . '_banner_' . $bannerImage->getClientOriginalName();
            $bannerImage->move(public_path('uploads/pages'), $bannerImageName);
            $validated['banner_image'] = '/uploads/pages/' . $bannerImageName;
        }

        // Remove file fields from validated data
        unset($validated['featured_image_file'], $validated['banner_image_file']);

        // Set author
        $validated['author_id'] = Auth::id();
        $validated['updated_by'] = Auth::id();

        // Handle checkbox
        $validated['is_featured'] = $request->has('is_featured') ? 1 : 0;

        $page = Page::create($validated);

        return redirect()->route('admin.pages.index')
            ->with('success', 'Page created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Page $page)
    {
        return view('admin.pages.show', compact('page'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Page $page)
    {
        return view('admin.pages.edit', compact('page'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Page $page)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:pages,slug,' . $page->id,
            'content' => 'required|string',
            'excerpt' => 'nullable|string',
            'status' => 'required|in:draft,published,private',
            'sort_order' => 'nullable|integer',
            'is_featured' => 'nullable|boolean',
            'published_at' => 'nullable|date',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
            'meta_keywords' => 'nullable|string',
            'meta_robots' => 'nullable|string|max:255',
            'canonical_url' => 'nullable|string|max:255',
            'schema_markup' => 'nullable|string',
            'featured_image' => 'nullable|string|max:255',
            'featured_image_file' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'featured_image_alt' => 'nullable|string|max:255',
            'featured_image_caption' => 'nullable|string|max:255',
            'banner_image' => 'nullable|string|max:255',
            'banner_image_file' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'banner_image_alt' => 'nullable|string|max:255',
        ]);

        // Generate slug if not provided
        if (empty($validated['slug'])) {
            $validated['slug'] = Str::slug($validated['title']);
        }

        // Handle featured image upload
        if ($request->hasFile('featured_image_file')) {
            // Delete old image if exists
            if ($page->featured_image && file_exists(public_path($page->featured_image))) {
                unlink(public_path($page->featured_image));
            }
            
            $featuredImage = $request->file('featured_image_file');
            $featuredImageName = time() . '_featured_' . $featuredImage->getClientOriginalName();
            $featuredImage->move(public_path('uploads/pages'), $featuredImageName);
            $validated['featured_image'] = '/uploads/pages/' . $featuredImageName;
        }

        // Handle banner image upload
        if ($request->hasFile('banner_image_file')) {
            // Delete old image if exists
            if ($page->banner_image && file_exists(public_path($page->banner_image))) {
                unlink(public_path($page->banner_image));
            }
            
            $bannerImage = $request->file('banner_image_file');
            $bannerImageName = time() . '_banner_' . $bannerImage->getClientOriginalName();
            $bannerImage->move(public_path('uploads/pages'), $bannerImageName);
            $validated['banner_image'] = '/uploads/pages/' . $bannerImageName;
        }

        // Remove file fields from validated data
        unset($validated['featured_image_file'], $validated['banner_image_file']);

        // Set updated by
        $validated['updated_by'] = Auth::id();

        // Handle checkbox
        $validated['is_featured'] = $request->has('is_featured') ? 1 : 0;

        $page->update($validated);

        return redirect()->route('admin.pages.index')
            ->with('success', 'Page updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Page $page)
    {
        $page->delete();

        return redirect()->route('admin.pages.index')
            ->with('success', 'Page deleted successfully.');
    }
}
