<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Reward;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class RewardController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $rewards = Reward::orderBy('sort_order')
                        ->orderBy('title')
                        ->paginate(10);
        return view('admin.rewards.index', compact('rewards'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.rewards.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|max:255',
            'slug' => 'nullable|max:255|unique:rewards',
            'content' => 'nullable',
            'excerpt' => 'nullable',
            'status' => 'required|in:draft,published,archived',
            'sort_order' => 'nullable|integer',
            'is_featured' => 'boolean',
            'published_at' => 'nullable|date',
            'meta_title' => 'nullable|max:255',
            'meta_description' => 'nullable',
            'meta_keywords' => 'nullable',
            'meta_robots' => 'nullable|max:255',
            'canonical_url' => 'nullable|max:255',
            'schema_markup' => 'nullable',
            'featured_image' => 'nullable|image|max:2048', // 2MB max
            'featured_image_alt' => 'nullable|max:255',
            'featured_image_caption' => 'nullable|max:255',
            'banner_image' => 'nullable|image|max:2048',
            'banner_image_alt' => 'nullable|max:255',
        ]);
        
        // Handle image uploads
        if ($request->hasFile('featured_image')) {
            $validated['featured_image'] = $request->file('featured_image')->store('rewards/featured', 'public');
        }
        
        if ($request->hasFile('banner_image')) {
            $validated['banner_image'] = $request->file('banner_image')->store('rewards/banners', 'public');
        }
        
        // Set the author
        $validated['author_id'] = Auth::id();
        $validated['updated_by'] = Auth::id();
        
        // Generate slug if not provided
        if (empty($validated['slug'])) {
            $validated['slug'] = Str::slug($validated['title']);
        }
        
        Reward::create($validated);
        
        return redirect()->route('admin.rewards.index')
            ->with('success', 'Reward created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Reward $reward)
    {
        return view('admin.rewards.show', compact('reward'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Reward $reward)
    {
        return view('admin.rewards.edit', compact('reward'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Reward $reward)
    {
        $validated = $request->validate([
            'title' => 'required|max:255',
            'slug' => 'nullable|max:255|unique:rewards,slug,'.$reward->id,
            'content' => 'nullable',
            'excerpt' => 'nullable',
            'status' => 'required|in:draft,published,archived',
            'sort_order' => 'nullable|integer',
            'is_featured' => 'boolean',
            'published_at' => 'nullable|date',
            'meta_title' => 'nullable|max:255',
            'meta_description' => 'nullable',
            'meta_keywords' => 'nullable',
            'meta_robots' => 'nullable|max:255',
            'canonical_url' => 'nullable|max:255',
            'schema_markup' => 'nullable',
            'featured_image' => 'nullable|image|max:2048', // 2MB max
            'featured_image_alt' => 'nullable|max:255',
            'featured_image_caption' => 'nullable|max:255',
            'banner_image' => 'nullable|image|max:2048',
            'banner_image_alt' => 'nullable|max:255',
        ]);
        
        // Handle image uploads
        if ($request->hasFile('featured_image')) {
            // Delete old image if exists
            if ($reward->featured_image) {
                Storage::disk('public')->delete($reward->featured_image);
            }
            $validated['featured_image'] = $request->file('featured_image')->store('rewards/featured', 'public');
        }
        
        if ($request->hasFile('banner_image')) {
            // Delete old image if exists
            if ($reward->banner_image) {
                Storage::disk('public')->delete($reward->banner_image);
            }
            $validated['banner_image'] = $request->file('banner_image')->store('rewards/banners', 'public');
        }
        
        // Set the updater
        $validated['updated_by'] = Auth::id();
        
        // Generate slug if not provided
        if (empty($validated['slug'])) {
            $validated['slug'] = Str::slug($validated['title']);
        }
        
        $reward->update($validated);
        
        return redirect()->route('admin.rewards.index')
            ->with('success', 'Reward updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Reward $reward)
    {
        // Delete associated images
        if ($reward->featured_image) {
            Storage::disk('public')->delete($reward->featured_image);
        }
        
        if ($reward->banner_image) {
            Storage::disk('public')->delete($reward->banner_image);
        }
        
        $reward->delete();
        
        return redirect()->route('admin.rewards.index')
            ->with('success', 'Reward deleted successfully.');
    }
}
