<?php

namespace App\Http\Controllers;

use App\Models\Slide;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class SlideController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Slide::query();

        // Search functionality
        if ($request->filled('search')) {
            $search = $request->get('search');
            $query->where(function ($q) use ($search) {
                $q->where('name', 'LIKE', "%{$search}%")
                  ->orWhere('description', 'LIKE', "%{$search}%");
            });
        }

        // Status filter
        if ($request->filled('status')) {
            $query->where('status', $request->get('status'));
        }

        $slides = $query->orderBy('sort_order', 'asc')
                       ->orderBy('created_at', 'desc')
                       ->paginate(10);

        return view('slides.index', compact('slides'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('slides.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'image' => 'required|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'image_alt' => 'nullable|string|max:255',
            'link_url' => 'nullable|url|max:255',
            'link_text' => 'nullable|string|max:100',
            'link_new_tab' => 'boolean',
            'status' => 'required|in:active,inactive',
            'sort_order' => 'required|integer|min:0',
        ]);

        $data = $request->all();
        
        // Handle checkbox for link_new_tab (ensure it's boolean)
        $data['link_new_tab'] = $request->has('link_new_tab');

        // Handle image upload
        if ($request->hasFile('image')) {
            \Log::info('Image file detected', ['file' => $request->file('image')->getClientOriginalName()]);
            $data['image'] = $request->file('image')->store('slides', 'public');
            \Log::info('Image stored at', ['path' => $data['image']]);
        } else {
            \Log::info('No image file detected in request');
        }

        Slide::create($data);

        return redirect()->route('admin.slides.index')
                        ->with('success', 'Slide created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Slide $slide)
    {
        return view('slides.show', compact('slide'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Slide $slide)
    {
        return view('slides.edit', compact('slide'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Slide $slide)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'image_alt' => 'nullable|string|max:255',
            'link_url' => 'nullable|url|max:255',
            'link_text' => 'nullable|string|max:100',
            'link_new_tab' => 'boolean',
            'status' => 'required|in:active,inactive',
            'sort_order' => 'required|integer|min:0',
        ]);

        $data = $request->all();
        
        // Handle checkbox for link_new_tab (ensure it's boolean)
        $data['link_new_tab'] = $request->has('link_new_tab');

        // Handle image upload
        if ($request->hasFile('image')) {
            \Log::info('Image file detected for update', ['file' => $request->file('image')->getClientOriginalName()]);
            // Delete old image
            if ($slide->image && Storage::disk('public')->exists($slide->image)) {
                Storage::disk('public')->delete($slide->image);
                \Log::info('Old image deleted', ['path' => $slide->image]);
            }
            $data['image'] = $request->file('image')->store('slides', 'public');
            \Log::info('New image stored at', ['path' => $data['image']]);
        } else {
            \Log::info('No image file detected in update request');
        }

        $slide->update($data);

        return redirect()->route('admin.slides.index')
                        ->with('success', 'Slide updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Slide $slide)
    {
        // Delete image file
        if ($slide->image && Storage::disk('public')->exists($slide->image)) {
            Storage::disk('public')->delete($slide->image);
        }

        $slide->delete();

        return redirect()->route('admin.slides.index')
                        ->with('success', 'Slide deleted successfully.');
    }
}
