<?php

namespace App\Http\Controllers\Membership;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Member;
use App\Models\Region;
use App\Models\Reward;
use App\Models\Point;
use App\Models\Tier;
use App\Models\Donation;
use App\Models\MembershipApiToken;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Session;
use Illuminate\Validation\Rules;

class MembershipController extends Controller
{
  /**
   * Third-party API configuration
   */
  private $apiBaseUrl;
  private $apiKey;
  private $apiTimeout;

  public function __construct()
  {
    $this->apiBaseUrl = env('MEMBERSHIP_API_URL');
    $this->apiKey = env('MEMBERSHIP_API_KEY');
    $this->apiTimeout = env('MEMBERSHIP_API_TIMEOUT', 30);
  }

  /**
   * Get or refresh API token
   */
  private function getApiToken()
  {
    // First, try to get a valid existing token
    $token = MembershipApiToken::getCurrentToken();

    if ($token && $token->isValid()) {
      $token->markAsUsed();
      return $token->access_token;
    }

    // If no valid token, request a new one
    return $this->requestNewToken();
  }

  /**
   * Request new API token from the membership service
   */
  private function requestNewToken()
  {
    try {
      $response = Http::timeout($this->apiTimeout)
        ->withHeaders([
          'Content-Type' => 'application/json',
          'x-api-key' => $this->apiKey
        ])
        ->post($this->apiBaseUrl . '/v1/login/', [
          'username' => 'LJIWebApi@thelalit.com',
          'password' => 'LJIWeb#2@3@34Nj2nwco'
        ]);

      if ($response->successful()) {
        $tokenData = $response->json();

        // Deactivate old tokens
        MembershipApiToken::where('is_active', true)->update(['is_active' => false]);

        // Store new token
        $tokenRecord = MembershipApiToken::createFromResponse($tokenData);

        return $tokenRecord->access_token;
      }

      Log::error('Failed to get API token', [
        'status' => $response->status(),
        'response' => $response->json()
      ]);

      return null;
    } catch (\Exception $e) {
      Log::error('API Token Request Error: ' . $e->getMessage());
      return null;
    }
  }

  /**
   * Clean up expired tokens (can be called via scheduled task)
   */
  public function cleanupExpiredTokens()
  {
    $deletedCount = MembershipApiToken::cleanupExpiredTokens();
    Log::info("Cleaned up {$deletedCount} expired API tokens");
    return $deletedCount;
  }

  /**
   * Make API request to third-party service
   */
  private function makeApiRequest($endpoint, $method = 'GET', $data = [], $headers = [])
  {
    try {
      // Get valid token for API requests
      $token = $this->getApiToken();

      if (!$token) {
        Log::error('No valid API token available for request', ['endpoint' => $endpoint]);
        return [
          'success' => false,
          'error' => 'Unable to authenticate with membership service',
          'status' => 401
        ];
      }

      $defaultHeaders = [
        'Authorization' => 'Bearer ' . $token,
        'Content-Type' => 'application/json',
        'Accept' => 'application/json',
      ];

      $headers = array_merge($defaultHeaders, $headers);
      $url = $this->apiBaseUrl . $endpoint;

      $httpClient = Http::withHeaders($headers)->timeout($this->apiTimeout);

      switch (strtoupper($method)) {
        case 'POST':
          $response = $httpClient->post($url, $data);
          break;
        case 'PUT':
          $response = $httpClient->put($url, $data);
          break;
        case 'PATCH':
          $response = $httpClient->patch($url, $data);
          break;
        case 'DELETE':
          $response = $httpClient->delete($url, $data);
          break;
        case 'GET':
        default:
          $response = $httpClient->get($url, $data);
          break;
      }

      if ($response->successful()) {
        return [
          'success' => true,
          'data' => $response->json(),
          'status' => $response->status()
        ];
      }

      return [
        'success' => false,
        'error' => $response->json()['message'] ?? 'API request failed',
        'status' => $response->status()
      ];
    } catch (\Exception $e) {
      Log::error('Membership API Error: ' . $e->getMessage(), [
        'endpoint' => $endpoint,
        'method' => $method,
        'data' => $data
      ]);

      return [
        'success' => false,
        'error' => 'Unable to connect to membership service',
        'status' => 500
      ];
    }
  }

  /**
   * Get authenticated user's membership ID for API calls
   */
  private function getMembershipId()
  {
    $membershipUser = Session::get('membership_user');
    if ($membershipUser) {
      return $membershipUser['membership_id'] ?? $membershipUser['membership_number'] ?? $membershipUser['email'];
    }

    // Fallback to authenticated user if no membership session
    $user = Auth::user();
    return $user->membership_id ?? $user->email ?? null;
  }

  /**
   * Get member ID from session (new session structure)
   */
  private function getMemberIdFromSession()
  {
    return Session::get('member_id');
  }

  /**
   * Get API member ID from session (legacy support)
   */
  private function getApiMemberIdFromSession()
  {
    $membershipUser = Session::get('membership_user');
    return $membershipUser['membership_id'] ?? null;
  }

  /**
   * Display the set new password page.
   */
  public function setNewPassword(Request $request)
  {
    // Get email from request or session
    $email = $request->get('email', '');

    // If no email in request, try to get from logged in member session
    if (empty($email)) {
      $memberId = Session::get('member_id');
      if ($memberId) {
        $member = Member::find($memberId);
        if ($member) {
          $email = $member->email;
        }
      }

      // Fallback to session email
      if (empty($email)) {
        $email = Session::get('member_email', '');
      }
    }

    return view('membership.set-new-password', [
      'email' => $email
    ]);
  }

  /**
   * Display the user's membership dashboard.
   */
  public function dashboard()
  {
    // Check if member is logged in via member login (new API)
    $memberId = Session::get('member_id');
    $membershipUser = null;

    if ($memberId) {
      // Get member data from database
      $member = \App\Models\Member::find($memberId);

      if ($member) {
        $membershipUser = [
          'id' => $member->id,
          'name' => $member->full_name,
          'first_name' => $member->first_name,
          'last_name' => $member->last_name,
          'email' => $member->email,
          'mobile' => $member->mobile,
          'membership_id' => $member->api_member_id,
          'membership_number' => $member->member_number,
          'tier' => $member->tier,
          'points' => $member->points,
          'status' => $member->status,
          'tier_data' => $member->tier_data,
          'balances' => $member->balances,
          'enrollment_date' => $member->enrollment_date,
          'last_login_at' => $member->last_login_at,
          'extra_data' => $member->extra_data,
        ];

        // Calculate total points from balances - get LaLiT Points balance
        $totalPoints = 0;
        if (is_array($member->balances)) {
          foreach ($member->balances as $balance) {
            if (isset($balance['loyalty_account']) && $balance['loyalty_account'] === 'LaLiT Points') {
              $totalPoints = (int) ($balance['balance'] ?? 0);
              break;
            }
          }
        }

        // Use points from balances or fallback to member points
        $totalPoints = $totalPoints > 0 ? $totalPoints : $member->points;
      } else {
        // Member not found, redirect to login
        return redirect()->route('home')->with('error', 'Please log in to access your dashboard');
      }
    } else {
      // Check old session format for backward compatibility
      $membershipUser = Session::get('membership_user');

      if (!$membershipUser) {
        return redirect()->route('home')->with('error', 'Please log in to access your dashboard');
      }

      // Set default points if not available
      $totalPoints = $membershipUser['points'] ?? 0;
    }

    $membershipId = $membershipUser['membership_id'] ?? $membershipUser['membership_number'];

    // // Get user data from API
    $userResponse = $this->makeApiRequest("/members/{$membershipId}");
    // if (!$userResponse['success']) {
    //     return redirect()->route('home')->with('error', 'Unable to load membership data');
    // }

    $memberData = $userResponse['data'] ?? [];

    // // Get user's current tier
    $tierResponse = $this->makeApiRequest("/members/{$membershipId}/tier");
    $currentTier = $tierResponse['success'] ? $tierResponse['data'] : null;

    // Get recent rewards
    $rewardsResponse = $this->makeApiRequest("/rewards/available", 'GET', ['limit' => 3]);
    $recentRewards = collect($rewardsResponse['success'] ? $rewardsResponse['data'] : []);

    // Get points summary
    $pointsResponse = $this->makeApiRequest("/members/{$membershipId}/points/summary");
    $pointsData = $pointsResponse['success'] ? $pointsResponse['data'] : [];

    // Priority order for total points:
    // 1. Session member_points (updated after donations/transactions)
    // 2. Member balances from database
    // 3. API response
    $sessionPoints = Session::get('member_points', 0);
    if ($sessionPoints > 0) {
      $totalPoints = $sessionPoints;
    } else {
      $totalPoints = $totalPoints > 0 ? $totalPoints : ($pointsData['current_balance'] ?? 0);
    }
    
    $pointsToNextTier = $pointsData['points_to_next_tier'] ?? 0;

    // Get recent activities
    $activitiesResponse = $this->makeApiRequest("/members/{$membershipId}/activities", 'GET', ['limit' => 5]);
    $recentActivities = $activitiesResponse['success'] ? $activitiesResponse['data'] : [];

    // Add tier points data
    $tierPointsCollected = 0; // Points collected in current tier

    return view('membership.dashboard', compact(
      'membershipUser',
      'memberData',
      'currentTier',
      'recentRewards',
      'totalPoints',
      'pointsToNextTier',
      'recentActivities',
      'tierPointsCollected'
    ));
  }

  /**
   * Display the edit profile form.
   */
  public function editProfile()
  {
    $user = Auth::user();
    $member = null;

    // Primary: Get member using member_id from session (new session structure)
    $memberId = $this->getMemberIdFromSession();

    if ($memberId) {
      // Try to get member data using member_id from session
      $member = Member::find($memberId);

      // Log for debugging
      Log::info('Looking for member with member_id from session: ' . $memberId, [
        'found' => $member ? true : false,
        'member_record' => $member ? $member->toArray() : null
      ]);
    }

    // Fallback 1: Try legacy API member ID approach
    if (!$member) {
      $apiMemberId = $this->getApiMemberIdFromSession();
      if ($apiMemberId) {
        $member = Member::where('api_member_id', $apiMemberId)->first();
        Log::info('Fallback: Looking for member with api_member_id: ' . $apiMemberId, [
          'found' => $member ? true : false
        ]);
      }
    }

    // Fallback 2: Try to find member by user_id or email
    if (!$member && $user) {
      $member = Member::where('user_id', $user->id)->first();

      if (!$member) {
        $member = Member::where('email', $user->email)->first();
      }

      Log::info('Fallback: Looking for member by user_id/email', [
        'user_id' => $user->id,
        'user_email' => $user->email,
        'found' => $member ? true : false
      ]);
    }

    // Get countries for dropdown
    $countries = \App\Models\Country::select('country_id', 'name')->orderBy('name')->get();

    // Log session data for debugging
    Log::info('Edit Profile Debug Session Data', [
      'member_id' => $memberId,
      'member_number' => Session::get('member_number'),
      'member_name' => Session::get('member_name'),
      'member_token' => Session::get('member_token') ? 'present' : 'missing',
      'user_id' => $user ? $user->id : null,
      'member_found' => $member ? true : false,
      'member_record_id' => $member ? $member->id : null
    ]);

    return view('membership.edit-profile', compact('user', 'member', 'countries'));
  }

  /**
   * Display the user's rewards page.
   */
  public function myRewards()
  {
    $user = Auth::user();

    // Static data for available rewards
    $availableRewards = collect([
      [
        'id' => 1,
        'title' => 'Complimentary Spa Treatment',
        'description' => 'Enjoy a relaxing 60-minute spa treatment at any LaLit property',
        'points_required' => 5000,
        'category' => 'spa',
        'location' => 'mumbai',
        'image' => null,
        'expiry_date' => '2025-12-31',
        'popular' => true,
      ],
      [
        'id' => 2,
        'title' => 'Fine Dining Experience',
        'description' => 'Three-course dinner for two at our signature restaurant',
        'points_required' => 8000,
        'category' => 'dining',
        'location' => 'delhi',
        'image' => null,
        'expiry_date' => '2025-11-30',
        'popular' => false,
      ],
      [
        'id' => 3,
        'title' => 'Room Upgrade',
        'description' => 'Complimentary upgrade to suite on your next stay',
        'points_required' => 3000,
        'category' => 'accommodation',
        'location' => 'goa',
        'image' => null,
        'expiry_date' => '2025-10-15',
        'popular' => true,
      ],
      [
        'id' => 4,
        'title' => 'Cooking Class Experience',
        'description' => 'Learn from our executive chef in a private cooking class',
        'points_required' => 6000,
        'category' => 'experiences',
        'location' => 'bengaluru',
        'image' => null,
        'expiry_date' => '2025-09-30',
        'popular' => false,
      ],
      [
        'id' => 5,
        'title' => 'Heritage Tour',
        'description' => 'Guided heritage tour of local attractions with lunch',
        'points_required' => 4000,
        'category' => 'experiences',
        'location' => 'jaipur',
        'image' => null,
        'expiry_date' => '2025-11-15',
        'popular' => true,
      ],
      [
        'id' => 6,
        'title' => 'Shopping Voucher',
        'description' => '₹2000 shopping voucher for our retail partners',
        'points_required' => 2500,
        'category' => 'retail',
        'location' => 'mumbai',
        'image' => null,
        'expiry_date' => '2025-12-31',
        'popular' => false,
      ]
    ]);

    // Static data for earned rewards
    $earnedRewards = [
      [
        'id' => 1,
        'title' => 'Welcome Drink Voucher',
        'description' => 'Complimentary welcome drink at arrival',
        'points_used' => 1000,
        'status' => 'available',
        'earned_at' => '2025-08-15',
        'expires_at' => '2025-12-15',
        'redeemed_at' => null,
      ],
      [
        'id' => 2,
        'title' => 'Late Checkout',
        'description' => 'Late checkout until 2:00 PM',
        'points_used' => 1500,
        'status' => 'redeemed',
        'earned_at' => '2025-07-20',
        'expires_at' => '2025-10-20',
        'redeemed_at' => '2025-08-10',
      ]
    ];

    // Static user points
    $userPoints = 12500;

    return view('membership.myrewards', compact(
      'user',
      'availableRewards',
      'earnedRewards',
      'userPoints'
    ));
  }

  /**
   * Display the user's statement page.
   */
  public function statement()
  {
    // Check if member is logged in via member login (new API)
    $memberId = Session::get('member_id');
    $membershipUser = null;
    $totalPoints = 0;
    $tierPoints = 0;
    $pointsDueToExpire = 0;
    $totalPointsExpired = 0;
    $pointsExpirationDetails = []; // Store expiration details with dates

    if ($memberId) {
      // Get member data from database
      $member = \App\Models\Member::find($memberId);
      if ($member) {
        $membershipUser = [
          'id' => $member->id,
          'name' => $member->full_name,
          'first_name' => $member->first_name,
          'last_name' => $member->last_name,
          'email' => $member->email,
          'mobile' => $member->mobile,
          'membership_id' => $member->api_member_id,
          'membership_number' => $member->member_number,
          'tier' => $member->tier,
          'points' => $member->points,
          'status' => $member->status,
          'tier_data' => $member->tier_data,
          'balances' => $member->balances,
          'enrollment_date' => $member->enrollment_date,
        ];

        // Fetch balances from API
        try {
          $apiController = new MembershipApiController();
          $balancesResponse = $apiController->getMemberBalances($member->member_number);

          if ($balancesResponse && isset($balancesResponse['balances'])) {
            $balances = $balancesResponse['balances'];
            $pointsExpiration = $balancesResponse['points_expiration'] ?? [];

            // Process balances - get LaLiT Points balance
            foreach ($balances as $balance) {
              if ($balance['loyalty_account'] === 'LaLiT Points') {
                $totalPoints = $balance['balance'] ?? 0;
                $totalPointsExpired = $balance['total_expired'] ?? 0;
                $tierPoints = $balance['balance'] ?? 0;
              }
            }

            // Calculate points due to expire - only for LaLiT Points
            foreach ($pointsExpiration as $expiration) {
              if (isset($expiration['loyalty_account']) && $expiration['loyalty_account'] === 'LaLiT Points') {
                $pointsDueToExpire += $expiration['points'] ?? 0;
                // Store expiration details for display
                $pointsExpirationDetails[] = [
                  'points' => $expiration['points'] ?? 0,
                  'expiration_date' => $expiration['expiration_date'] ?? null,
                  'reward_tag' => $expiration['reward_tag'] ?? null
                ];
              }
            }
          }
        } catch (\Exception $e) {
          Log::error('Failed to fetch member balances: ' . $e->getMessage());
          // Fallback to stored data - get LaLiT Points balance
          if (is_array($member->balances)) {
            foreach ($member->balances as $balance) {
              if (isset($balance['loyalty_account']) && $balance['loyalty_account'] === 'LaLiT Points') {
                $totalPoints = (int) ($balance['balance'] ?? 0);
                $tierPoints = $totalPoints;
                break;
              }
            }
          }
          // Final fallback to member points
          if ($totalPoints === 0) {
            $totalPoints = $member->points;
            $tierPoints = $totalPoints;
          }
        }

      } else {
        // Member not found, redirect to login
        return redirect()->route('home')->with('error', 'Please log in to access your statement');
      }
    } else {
      // Check old session format for backward compatibility
      $membershipUser = Session::get('membership_user');
      if (!$membershipUser) {
        return redirect()->route('home')->with('error', 'Please log in to access your statement');
      }
      // Set default points if not available
      $totalPoints = $membershipUser['points'] ?? 0;
      $tierPoints = $totalPoints;
    }

    return view('membership.statement', compact(
      'membershipUser',
      'totalPoints',
      'tierPoints',
      'pointsDueToExpire',
      'totalPointsExpired',
      'pointsExpirationDetails'
    ));
  }

  /**
   * Redeem a reward via API (AJAX endpoint).
   */
  public function redeemReward(Request $request)
  {
    $request->validate([
      'reward_id' => 'required|string'
    ]);

    $membershipId = $this->getMembershipId();
    $rewardId = $request->input('reward_id');

    $redemptionData = [
      'reward_id' => $rewardId,
      'member_id' => $membershipId
    ];

    $response = $this->makeApiRequest("/members/{$membershipId}/rewards/redeem", 'POST', $redemptionData);

    if ($response['success']) {
      $data = $response['data'];
      return response()->json([
        'success' => true,
        'message' => 'Reward redeemed successfully!',
        'remaining_points' => $data['remaining_points'] ?? 0,
        'redemption_code' => $data['redemption_code'] ?? null
      ]);
    }

    return response()->json([
      'success' => false,
      'message' => $response['error'] ?? 'Failed to redeem reward'
    ], 400);
  }

  /**
   * Handle member login via API
   */
  public function login(Request $request)
  {
    $request->validate([
      'membership_number' => 'required|string',
      'password' => 'required|string',
    ]);

    $loginData = [
      'membership_number' => $request->membership_number,
      'password' => $request->password,
    ];

    $response = $this->makeApiRequest('/auth/login', 'POST', $loginData);
    // generate dummy response for testing
    // $response = [
    //   'success' => true,
    //   'message' => 'Login successful',
    //   'data' => [
    //     'membership_id' => '123456',
    //     'membership_number' => $request->membership_number,
    //     'name' => 'John Doe',
    //     'email' => 'john.doe@example.com',
    //     'api_token' => 'dummy_api_token'
    //   ]
    // ];

    if ($response['success']) {
      // Store membership token and user data in session
      Session::put('membership_token', $response['data']['api_token']);
      Session::put('membership_user', $response['data']);
      Session::put('membership_authenticated', true);

      // Debug log
      Log::info('Membership login successful', [
        'user' => $response['data'],
        'is_ajax' => $request->ajax(),
        'expects_json' => $request->expectsJson(),
        'redirect_route' => route('membership.dashboard')
      ]);

      // Return JSON response for AJAX requests
      if ($request->expectsJson() || $request->ajax()) {
        return response()->json([
          'success' => true,
          'message' => 'Login successful',
          'data' => [
            'user' => $response['data'],
            'token' => $response['data']['api_token']
          ]
        ]);
      }

      // For non-AJAX requests, redirect to dashboard
      return redirect()->route('membership.dashboard');
    }

    // Return JSON response for AJAX requests
    if ($request->expectsJson() || $request->ajax()) {
      return response()->json([
        'success' => false,
        'message' => $response['error'] ?? 'Invalid credentials'
      ], 400);
    }

    return back()->withErrors([
      'membership_number' => $response['error'] ?? 'Invalid credentials',
    ])->onlyInput('membership_number');
  }

  /**
   * Handle forgot password via API
   */
  public function forgotPassword(Request $request)
  {
    $request->validate([
      'email' => 'required|email',
    ]);

    $response = $this->makeApiRequest('/auth/forgot-password', 'POST', [
      'email' => $request->email
    ]);

    if ($response['success']) {
      // Return JSON response for AJAX requests
      if ($request->expectsJson() || $request->ajax()) {
        return response()->json([
          'success' => true,
          'message' => 'Password reset link sent to your email!'
        ]);
      }

      return back()->with('status', 'Password reset link sent to your email!');
    }

    // Return JSON response for AJAX requests
    if ($request->expectsJson() || $request->ajax()) {
      return response()->json([
        'success' => false,
        'message' => $response['error'] ?? 'Unable to send reset link'
      ], 400);
    }

    return back()->withErrors([
      'email' => $response['error'] ?? 'Unable to send reset link',
    ]);
  }

  /**
   * Handle password reset via API
   */
  public function resetPassword(Request $request)
  {
    $request->validate([
      'token' => 'required',
      'email' => 'required|email',
      'password' => 'required|string|min:8|confirmed',
    ]);

    $resetData = [
      'token' => $request->token,
      'email' => $request->email,
      'password' => $request->password,
      'password_confirmation' => $request->password_confirmation,
    ];

    $response = $this->makeApiRequest('/auth/reset-password', 'POST', $resetData);

    if ($response['success']) {
      return redirect()->route('home')->with('status', 'Password reset successful! You can now login.');
    }

    return back()->withErrors([
      'email' => $response['error'] ?? 'Password reset failed',
    ])->withInput();
  }

  /**
   * Handle logout
   */
  public function logout(Request $request)
  {
    $membershipId = $this->getMembershipId();

    // Notify API about logout
    $this->makeApiRequest("/auth/logout", 'POST', ['member_id' => $membershipId]);

    // Clear membership session data
    Session::forget(['membership_token', 'membership_user', 'membership_authenticated', 'api_token']);

    Auth::logout();

    // Return JSON response for AJAX requests
    if ($request->expectsJson() || $request->ajax()) {
      return response()->json([
        'success' => true,
        'message' => 'Logged out successfully'
      ]);
    }

    return redirect()->route('home')->with('success', 'Logged out successfully');
  }

  /**
   * Get dashboard data via AJAX
   */
  public function getDashboardData(Request $request)
  {
    $membershipId = $this->getMembershipId();

    try {
      // Get member profile data
      $profileResponse = $this->makeApiRequest("/members/{$membershipId}/profile");

      // Get rewards data
      $rewardsResponse = $this->makeApiRequest("/members/{$membershipId}/rewards");

      // Get recent activities
      $activitiesResponse = $this->makeApiRequest("/members/{$membershipId}/activities", 'GET', ['limit' => 10]);

      return response()->json([
        'success' => true,
        'data' => [
          'totalPoints' => $profileResponse['data']['total_points'] ?? 0,
          'pointsToNextTier' => $profileResponse['data']['points_to_next_tier'] ?? 0,
          'availableRewards' => $rewardsResponse['data']['available_rewards'] ?? [],
          'recentActivities' => $activitiesResponse['data']['activities'] ?? [],
          'tierProgress' => $profileResponse['data']['tier_progress'] ?? null
        ]
      ]);
    } catch (\Exception $e) {
      Log::error('Failed to get dashboard data: ' . $e->getMessage());

      return response()->json([
        'success' => false,
        'message' => 'Failed to load dashboard data'
      ], 500);
    }
  }

  /**
   * Get rewards data via AJAX
   */
  public function getRewardsData(Request $request)
  {
    $membershipId = $this->getMembershipId();

    try {
      // Get available rewards
      $availableResponse = $this->makeApiRequest("/members/{$membershipId}/rewards/available");

      // Get earned/redeemed rewards
      $earnedResponse = $this->makeApiRequest("/members/{$membershipId}/rewards/earned");

      // Get user points
      $profileResponse = $this->makeApiRequest("/members/{$membershipId}/profile");

      return response()->json([
        'success' => true,
        'data' => [
          'availableRewards' => $availableResponse['data']['rewards'] ?? [],
          'earnedRewards' => $earnedResponse['data']['rewards'] ?? [],
          'userPoints' => $profileResponse['data']['total_points'] ?? 0
        ]
      ]);
    } catch (\Exception $e) {
      Log::error('Failed to get rewards data: ' . $e->getMessage());

      return response()->json([
        'success' => false,
        'message' => 'Failed to load rewards data'
      ], 500);
    }
  }

  /**
   * Get statement data via AJAX
   */
  public function getStatementData(Request $request)
  {
    $membershipId = $this->getMembershipId();

    $filters = [
      'from_date' => $request->get('from_date'),
      'to_date' => $request->get('to_date'),
      'type' => $request->get('type'),
      'page' => $request->get('page', 1),
      'per_page' => $request->get('per_page', 20)
    ];

    // Remove null filters
    $filters = array_filter($filters, function ($value) {
      return $value !== null && $value !== '';
    });

    try {
      // Get transaction history
      $transactionsResponse = $this->makeApiRequest("/members/{$membershipId}/transactions", 'GET', $filters);

      // Get summary data
      $summaryResponse = $this->makeApiRequest("/members/{$membershipId}/statement/summary", 'GET', $filters);

      return response()->json([
        'success' => true,
        'data' => [
          'transactions' => $transactionsResponse['data']['transactions'] ?? [],
          'currentBalance' => $summaryResponse['data']['current_balance'] ?? 0,
          'totalEarned' => $summaryResponse['data']['total_earned'] ?? 0,
          'totalRedeemed' => $summaryResponse['data']['total_redeemed'] ?? 0,
          'monthlySummary' => $summaryResponse['data']['monthly_summary'] ?? [],
          'has_more' => $transactionsResponse['data']['has_more'] ?? false
        ]
      ]);
    } catch (\Exception $e) {
      Log::error('Failed to get statement data: ' . $e->getMessage());

      return response()->json([
        'success' => false,
        'message' => 'Failed to load statement data'
      ], 500);
    }
  }

  /**
   * Update profile
   */
  public function updateProfile(Request $request)
  {
    $request->validate([
      'title' => 'nullable|string|max:10',
      'first_name' => 'required|string|max:255',
      'last_name' => 'required|string|max:255',
      'middle_name' => 'nullable|string|max:255',
      'email' => 'required|email|max:255',
      'mobile' => 'nullable|string|max:20',
      'date_of_birth' => 'nullable|date',
      'gender' => 'nullable|string|in:male,female,other',
      'nationality' => 'nullable|string|max:100',
      'marriage_anniversary' => 'nullable|date',
      'company' => 'nullable|string|max:255',
      'designation' => 'nullable|string|max:50',
      'address_line_1' => 'nullable|string|max:255',
      'address_line_2' => 'nullable|string|max:255',
      'state' => 'nullable|string|max:100',
      'city' => 'nullable|string|max:100',
      'postal_code' => 'nullable|digits:6',
      'country' => 'nullable|string|max:100',
    ]);

    try {
      $user = Auth::user();

      // Update user record if user is authenticated
      if ($user) {
        $user->update([
          'name' => $request->first_name . ' ' . $request->last_name,
          'email' => $request->email,
        ]);
      }

      // Primary: Get member using member_id from session (new session structure)
      $memberId = $this->getMemberIdFromSession();
      $member = null;

      if ($memberId) {
        $member = Member::find($memberId);
      }

      // Fallback 1: Try legacy API member ID approach
      if (!$member) {
        $apiMemberId = $this->getApiMemberIdFromSession();
        if ($apiMemberId) {
          $member = Member::where('api_member_id', $apiMemberId)->first();
        }
      }

      // Fallback 2: Find member record using user_id or email (if user is authenticated)
      if (!$member && $user) {
        $member = Member::where('user_id', $user->id)->first();
        if (!$member) {
          $member = Member::where('email', $user->email)->first();
        }
      }

      $memberData = [
        'user_id' => $user ? $user->id : null,
        'api_member_id' => $member ? $member->api_member_id : null,
        'title' => $request->title,
        'first_name' => $request->first_name,
        'middle_name' => $request->middle_name,
        'last_name' => $request->last_name,
        'email' => $request->email,
        'mobile' => $member->mobile,
        'date_of_birth' => $request->date_of_birth,
        'gender' => $request->gender,
        'nationality' => $request->nationality,
        'marriage_anniversary' => $request->marriage_anniversary,
        'company' => $request->company,
        'designation' => $request->designation,
        'address_line1' => $request->address_line_1,
        'address_line2' => $request->address_line_2,
        'city' => $request->city,
        'state' => $request->state,
        'country' => $request->country,
        'postal_code' => $request->postal_code,
      ];

      // Log the update process
      Log::info('Updating member profile', [
        'member_id_from_session' => $memberId,
        'member_found' => $member ? true : false,
        'member_record_id' => $member ? $member->id : null,
        'operation' => $member ? 'update' : 'create'
      ]);

      // Check if member has member_number for API call
      if ($member && $member->member_number) {
        // Make API call to update member data using API controller
        $apiController = new \App\Http\Controllers\Membership\MembershipApiController();
        $apiSuccess = $apiController->updateMemberViaAPI($member, $request);

        if ($apiSuccess) {
          // Update local database after successful API call
          $member->update($memberData);
          
          // Update extra_data with additional fields
          $extraData = is_string($member->extra_data) ? json_decode($member->extra_data, true) : (array) ($member->extra_data ?? []);
          
          if ($request->marriage_anniversary) {
            $extraData['wedding_anniversary'] = $request->marriage_anniversary;
          }
          if ($request->company) {
            $extraData['company'] = $request->company;
          }
          if ($request->designation) {
            $extraData['job_title'] = $request->designation;
          }
          
          $member->update(['extra_data' => $extraData]);
          
          Log::info('Member profile updated successfully (API + DB)', ['member_id' => $member->id]);
          return redirect()->back()->with('status', 'Profile updated successfully!');
        } else {
          // API failed, but still update local database
          $member->update($memberData);
          Log::warning('API update failed, but local database updated', ['member_id' => $member->id]);
          return redirect()->back()->with('status', 'Profile updated locally (API update failed).');
        }
      } else {
        // No member or no member number, just update/create locally
        if ($member) {
          $member->update($memberData);
          Log::info('Member profile updated successfully (DB only)', ['member_id' => $member->id]);
        } else {
          $member = Member::create($memberData);
          // Update session with new member ID
          Session::put('member_id', $member->id);
          Session::put('member_number', $member->member_number);
          Session::put('member_name', $member->full_name);
          Log::info('New member profile created', ['member_id' => $member->id]);
        }
        return redirect()->back()->with('status', 'Profile updated successfully!');
      }

    } catch (\Exception $e) {
      Log::error('Failed to update profile: ' . $e->getMessage());

      return redirect()->back()->with('error', 'Failed to update profile. Please try again.');
    }
  }

  /**
   * Update Preferences via AJAX
   */
  public function updatePreferences(Request $request)
  {
    $request->validate([
      'traveller_type' => 'nullable|string|max:50',
      'frequent_cities' => 'nullable|array',
      'frequent_cities.*' => 'nullable|string|max:100',
      'room_preference' => 'nullable|array',
      'room_preference.*' => 'nullable|string|max:50',
      'beverage_preference' => 'nullable|string|max:50',
      'favourite_cuisine' => 'nullable|array',
      'favourite_cuisine.*' => 'nullable|string|max:100',
      'pillow_preference' => 'nullable|array',
      'pillow_preference.*' => 'nullable|string|max:100',
      'room_should_be' => 'nullable|array',
      'room_should_be.*' => 'nullable|string|max:100',
      'dietary_preference' => 'nullable|array',
      'dietary_preference.*' => 'nullable|string|max:100',
      'allergies' => 'nullable|array',
      'allergies.*' => 'nullable|string|max:100',
      'spice_level' => 'nullable|string|max:50',
    ]);

    // Get member from session
    $memberId = $this->getMemberIdFromSession();
    $member = Member::find($memberId);  

    if (!$member) {
      return response()->json([
        'success' => false,
        'message' => 'Member not found'
      ], 404);
    } 

    // Prepare preferences data - map form fields to API fields
    $preferencesData = [];
    
    // Traveller type
    if ($request->filled('traveller_type')) {
      $preferencesData['traveller'] = $request->traveller_type;
    }
    
    // Frequent cities - pass as array
    if ($request->has('frequent_cities')) {
      $preferencesData['frequent_city'] = $request->frequent_cities ?? [];
    }
    
    // Room preference - pass as array
    if ($request->has('room_preference')) {
      $preferencesData['room'] = $request->room_preference ?? [];
    }
    
    // Beverage preference
    if ($request->filled('beverage_preference')) {
      $preferencesData['beverage_preference'] = $request->beverage_preference;
    }
    
    // Favourite cuisine - pass as array
    if ($request->has('favourite_cuisine')) {
      $preferencesData['favorite_food'] = $request->favourite_cuisine ?? [];
    }
    
    // Pillow preference - pass as array
    if ($request->has('pillow_preference')) {
      $preferencesData['pillow'] = $request->pillow_preference ?? [];
    }
    
    // Room should be (floor preference) - pass as array
    if ($request->has('room_should_be')) {
      $preferencesData['floor'] = $request->room_should_be ?? [];
    }
    
    // Dietary preference - pass as array
    if ($request->has('dietary_preference')) {
      $preferencesData['dietary_preference'] = $request->dietary_preference ?? [];
    }
    
    // Allergies - pass as array
    if ($request->has('allergies')) {
      $preferencesData['allergy'] = $request->allergies ?? [];
    }
    
    // Spice level
    if ($request->filled('spice_level')) {
      $preferencesData['spice'] = $request->spice_level;
    }
    
    $preferencesData['enrolling_sponsor'] = $member->enrolling_sponsor ?? 1;

    try {
      // Get valid token for API request
      $token = $this->getApiToken();

      if (!$token) {
        Log::error('No valid API token available for preferences update');
        return response()->json([
          'success' => false,
          'message' => 'Unable to authenticate with membership service'
        ], 401);
      }

      // Make direct HTTP request with custom headers for JWT authentication
      $url = $this->apiBaseUrl . "/v1/members/update/{$member->member_number}";
      
      $response = Http::withHeaders([
        'Authorization' => 'JWT ' . $token,
        'x-api-key' => $this->apiKey,
        'Content-Type' => 'application/json',
        'Accept' => 'application/json',
      ])
      ->timeout($this->apiTimeout)
      ->put($url, $preferencesData);

      // Log the request and response
      Log::info('Member preferences update request', [
        'member_id' => $member->id,
        'member_number' => $member->member_number,
        'url' => $url,
        'data' => $preferencesData,
        'response_status' => $response->status(),
        'response_body' => $response->json()
      ]);

      if ($response->successful()) {
        // Update local member record with preferences data using same keys as API
        $extraData = $member->extra_data ? (is_string($member->extra_data) ? json_decode($member->extra_data, true) : $member->extra_data) : [];
        
        // Update extra_data with new preference values using API keys
        if ($request->filled('traveller_type')) {
          $extraData['traveller'] = $request->traveller_type;
        }
        if ($request->has('frequent_cities')) {
          $extraData['frequent_city'] = $request->frequent_cities;
        }
        if ($request->has('room_preference')) {
          $extraData['room'] = $request->room_preference;
        }
        if ($request->filled('beverage_preference')) {
          $extraData['beverage_preference'] = $request->beverage_preference;
        }
        if ($request->has('favourite_cuisine')) {
          $extraData['favorite_food'] = $request->favourite_cuisine;
        }
        if ($request->has('pillow_preference')) {
          $extraData['pillow'] = $request->pillow_preference;
        }
        if ($request->has('room_should_be')) {
          $extraData['floor'] = $request->room_should_be;
        }
        if ($request->has('dietary_preference')) {
          $extraData['dietary_preference'] = $request->dietary_preference;
        }
        if ($request->has('allergies')) {
          $extraData['allergy'] = $request->allergies;
        }
        if ($request->filled('spice_level')) {
          $extraData['spice'] = $request->spice_level;
        }
        
        // Save updated extra_data to member record
        $member->update([
          'extra_data' => $extraData
        ]);

        Log::info('Member preferences updated locally', [
          'member_id' => $member->id,
          'preferences' => $extraData
        ]);

        return response()->json([
          'success' => true,
          'message' => 'Preferences details updated successfully'
        ]);
      }

      // Handle API error response
      $errorMessage = 'Failed to update preferences';
      $responseData = $response->json();
      
      if (isset($responseData['message'])) {
        $errorMessage = $responseData['message'];
      }

      Log::error('Failed to update member preferences', [
        'member_id' => $member->id,
        'status' => $response->status(),
        'response' => $responseData
      ]);

      return response()->json([
        'success' => false,
        'message' => $errorMessage
      ], $response->status());

    } catch (\Exception $e) {
      Log::error('Exception while updating preferences', [
        'member_id' => $member->id,
        'error' => $e->getMessage(),
        'trace' => $e->getTraceAsString()
      ]);

      return response()->json([
        'success' => false,
        'message' => 'An error occurred while updating preferences. Please try again.'
      ], 500);
    }
  }

  /**
   * Update password via AJAX
   */
  public function updatePassword(Request $request)
  {
    $request->validate([
      'current_password' => 'required|string',
      'password' => 'required|string|min:8|confirmed',
    ]);

    $membershipId = $this->getMembershipId();

    $passwordData = [
      'current_password' => $request->current_password,
      'new_password' => $request->password,
    ];

    try {
      $response = $this->makeApiRequest("/members/{$membershipId}/password", 'PUT', $passwordData);

      if ($response['success']) {
        return response()->json([
          'success' => true,
          'message' => 'Password updated successfully'
        ]);
      }

      return response()->json([
        'success' => false,
        'message' => $response['error'] ?? 'Failed to update password'
      ], 400);
    } catch (\Exception $e) {
      Log::error('Failed to update password: ' . $e->getMessage());

      return response()->json([
        'success' => false,
        'message' => 'Failed to update password'
      ], 500);
    }
  }

  /**
   * Display the donations page
   */
  public function donations()
  {
    try {
      // Get member info from session
      $memberId = Session::get('member_id');
      $memberApiId = Session::get('member_api_id');
      
      // Initialize member points
      $memberPoints = 0;
      
      // Try to fetch current points from member record and API
      if ($memberId) {
        // First, try to get from database member record
        $member = Member::find($memberId);
        
        if ($member) {
          // Try to get from member's balances data
          if (is_array($member->balances)) {
            foreach ($member->balances as $balance) {
              if (isset($balance['loyalty_account']) && $balance['loyalty_account'] === 'LaLiT Points') {
                $memberPoints = (int) ($balance['balance'] ?? 0);
                break;
              }
            }
          }
          
          // If still 0, try member points directly
          if ($memberPoints === 0) {
            $memberPoints = $member->points ?? 0;
          }
          
          // Try to fetch fresh data from API if we have member number
          if ($member->member_number && $memberPoints === 0) {
            try {
              $apiController = new MembershipApiController();
              $balancesResponse = $apiController->getMemberBalances($member->member_number);
              
              if ($balancesResponse && isset($balancesResponse['balances'])) {
                foreach ($balancesResponse['balances'] as $balance) {
                  if (isset($balance['loyalty_account']) && $balance['loyalty_account'] === 'LaLiT Points') {
                    $memberPoints = (int) ($balance['balance'] ?? 0);
                    break;
                  }
                }
              }
            } catch (\Exception $e) {
              Log::warning('Failed to fetch API balances for donations: ' . $e->getMessage());
            }
          }
        }
      }
      
      // Final fallback to session points
      if ($memberPoints === 0) {
        $memberPoints = Session::get('member_points', 0);
      }

      // Log the points for debugging
      Log::info('Donations page loaded', [
        'member_id' => $memberId,
        'member_points' => $memberPoints,
        'session_points' => Session::get('member_points', 0)
      ]);

      // Fetch active donation campaigns from database with images
      $donations = Donation::with('donationImage')
        ->where('is_active', true)
        ->ordered()
        ->get();

      return view('membership.donation', [
        'pageTitle' => 'Donations',
        'donations' => $donations,
        'memberPoints' => $memberPoints
      ]);
    } catch (\Exception $e) {
      Log::error('Failed to load donations page: ' . $e->getMessage());
      return redirect()->route('membership.dashboard')->with('error', 'Unable to load donations page.');
    }
  }

  /**
   * Display the participating hotels page
   */
  public function participatingHotels()
  {
    try {
      // Fetch active participating hotels from database
      $hotels = \App\Models\ParticipatingHotel::active()
        ->ordered()
        ->get();

      return view('membership.participating_hotels', [
        'pageTitle' => 'Participating Hotels',
        'hotels' => $hotels
      ]);
    } catch (\Exception $e) {
      Log::error('Failed to load participating hotels page: ' . $e->getMessage());
      return redirect()->route('membership.dashboard')->with('error', 'Unable to load participating hotels page.');
    }
  }

  /**
   * Display the rewards page
   */
  public function rewards()
  {
    try {
      // You can add logic here to fetch rewards data from API or database
      // For now, returning the view with static data (as shown in the blade template)
      return view('membership.rewards', [
        'pageTitle' => 'Rewards'
      ]);
    } catch (\Exception $e) {
      Log::error('Failed to load rewards page: ' . $e->getMessage());
      return redirect()->route('membership.dashboard')->with('error', 'Unable to load rewards page.');
    }
  }

  /**
   * Get regions for a specific country
   */
  public function getRegions($countryId)
  {
    try {
      $regions = Region::where('country_id', $countryId)
        ->select('id', 'region_id', 'name', 'region_code')
        ->orderBy('name')
        ->get();

      return response()->json([
        'success' => true,
        'data' => $regions
      ]);
    } catch (\Exception $e) {
      Log::error('Failed to fetch regions: ' . $e->getMessage());
      return response()->json([
        'success' => false,
        'message' => 'Failed to fetch regions'
      ], 500);
    }
  }

  /**
   * Get cities for a specific region using external API
   */
  /**
   * Get cities for a specific region using MembershipApiController
   */
  public function getCities($regionId)
  {
    try {
      // Delegate to MembershipApiController
      $apiController = new MembershipApiController();
      return $apiController->getCities($regionId);
    } catch (\Exception $e) {
      Log::error('Failed to get cities via API controller: ' . $e->getMessage());
      return response()->json([
        'success' => false,
        'message' => 'Failed to fetch cities'
      ], 500);
    }
  }
}
