<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Donation extends Model
{
  use HasFactory;

  protected $fillable = [
    'title',
    'image',
    'description',
    'donate_points',
    'donate_points_inr_value',
    'donation_link',
    'is_active'
  ];

  protected $casts = [
    'is_active' => 'boolean',
    'donate_points_inr_value' => 'decimal:2'
  ];

  /**
   * Scope to get only active donations
   */
  public function scopeActive($query)
  {
    return $query->where('is_active', true);
  }

  /**
   * Scope to get donations ordered by title
   */
  public function scopeOrdered($query)
  {
    return $query->orderBy('title', 'asc');
  }

  /**
   * Get the donation image associated with this donation.
   */
  public function donationImage()
  {
    return $this->hasOne(DonationImage::class, 'asset_image_id', 'image');
  }

  /**
   * Get formatted INR value
   * Formula: 1.25 donate points = 1 INR
   */
  public function getFormattedInrValueAttribute()
  {
    $inrValue = $this->donate_points / 1.25;
    return '₹' . number_format($inrValue, 2);
  }
}
