<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Support\Str;

class Point extends Model
{
    use HasFactory;      protected $fillable = [
        'title',
        'slug',
        'content',
        'excerpt',
        'hotel_stays_content',
        'hotel_stays_icon',
        'hotel_stays_title',
        'hotel_stays_link',
        'dining_content',
        'dining_icon',
        'dining_title',
        'dining_link',
        'spa_content',
        'spa_icon',
        'spa_title',
        'spa_link',
        'points_earning_image',
        'points_earning_image_alt',
        'status',
        'sort_order',
        'is_featured',
        'published_at',
        'meta_title',
        'meta_description',
        'meta_keywords',
        'meta_robots',
        'canonical_url',
        'og_data',
        'twitter_data',
        'schema_markup',
        'featured_image',
        'featured_image_alt',
        'featured_image_caption',
        'featured_image_meta',
        'gallery_images',
        'banner_image',
        'banner_image_alt',
        'author_id',
        'updated_by',
    ];

    protected $casts = [
        'published_at' => 'datetime',
        'is_featured' => 'boolean',
        'og_data' => 'array',
        'twitter_data' => 'array',
        'featured_image_meta' => 'array',
        'gallery_images' => 'array',
    ];
    
    /**
     * Get the route key for the model.
     */
    public function getRouteKeyName(): string
    {
        return 'slug';
    }
    
    /**
     * Get the author of the point.
     */
    public function author(): BelongsTo
    {
        return $this->belongsTo(User::class, 'author_id');
    }
    
    /**
     * Get the user who last updated the point.
     */
    public function updatedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'updated_by');
    }
    
    /**
     * Scope a query to only include published points.
     */
    public function scopePublished($query)
    {
        return $query->where('status', 'published')
                    ->where(function($q) {
                        $q->whereNull('published_at')
                          ->orWhere('published_at', '<=', now());
                    });
    }
    
    /**
     * Scope a query to only include featured points.
     */
    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }
    
    /**
     * Generate a unique slug for the point.
     */
    public static function generateUniqueSlug($title)
    {
        $slug = Str::slug($title);
        $count = static::whereRaw("slug RLIKE '^{$slug}(-[0-9]+)?$'")->count();
        
        return $count ? "{$slug}-{$count}" : $slug;
    }
    
    /**
     * Get the URL for the featured image.
     */
    public function getFeaturedImageUrlAttribute()
    {
        return $this->featured_image ? asset('storage/' . $this->featured_image) : null;
    }
    
    /**
     * Get the URL for the banner image.
     */
    public function getBannerImageUrlAttribute()
    {
        return $this->banner_image ? asset('storage/' . $this->banner_image) : null;
    }
    
    /**
     * Get the public URL for this point.
     */
    public function getUrlAttribute()
    {
        return route('pages.points', $this->slug);
    }
}
