/**
 * LaLiT Membership JavaScript Module
 * Handles all membership-related AJAX operations
 * Version: 1.0.0
 */

class LalitMembership {
    constructor() {
        this.config = {
            baseUrl: "/membership",
            apiUrl: "/api/membership",
            timeout: 30000,
            retryAttempts: 3,
            cache: new Map(),
            cacheTTL: 300000, // 5 minutes
        };

        this.state = {
            isLoggedIn: false,
            currentUser: null,  
            apiToken: null,
            loading: false,
            expiryWarningShown: false,
        };

        this.init();
    }

    /**
     * Initialize the membership module
     */
    init() {
        this.setupCSRFToken();
        this.bindEvents();
        this.checkAuthStatus();
        this.startAuthStatusMonitoring();
        console.log("LaLiT Membership module initialized");
    }

    /**
     * Setup CSRF token for all AJAX requests
     */
    setupCSRFToken() {
        const token = document
            .querySelector('meta[name="csrf-token"]')
            ?.getAttribute("content");
        if (token) {
            $.ajaxSetup({
                headers: {
                    "X-CSRF-TOKEN": token,
                },
            });
        }
    }

    /**
     * Bind event listeners
     */
    bindEvents() {
        // Unbind any existing handlers first to prevent duplicates
        $(document).off("submit", "#loginForm");
        $(document).off("input", "#membership_number, #loginPassword");
        $(document).off("submit", "#registerForm");
        $(document).off("submit", "#forgotPasswordForm");
        $(document).off("submit", "#profileUpdateForm");
        $(document).off("submit", "#passwordChangeForm");
        $(document).off("click", ".redeem-reward-btn");
        $(document).off("click", ".logout-btn");
        $(document).off("click", ".refresh-dashboard");
        $(document).off("click", "#filterStatement");
        $(document).off("click", "#loadMoreTransactions");
        
        // Login form submission
        $(document).on("submit", "#loginForm", (e) => this.handleLogin(e));

        // Clear login errors when user starts typing
        $(document).on("input", "#membership_number, #loginPassword", () => {
            this.hideLoginError();
        });

        // Registration form submission
        $(document).on("submit", "#registerForm", (e) =>
            this.handleRegistration(e)
        );

        // Forgot password form submission
        $(document).on("submit", "#forgotPasswordForm", (e) =>
            this.handleForgotPassword(e)
        );

        // Profile update form submission
        $(document).on("submit", "#profileUpdateForm", (e) =>
            this.handleProfileUpdate(e)
        );

        // Password change form submission
        $(document).on("submit", "#passwordChangeForm", (e) =>
            this.handlePasswordChange(e)
        );

        // Reward redemption
        $(document).on("click", ".redeem-reward-btn", (e) =>
            this.handleRewardRedemption(e)
        );

        // Logout
        $(document).on("click", ".logout-btn", (e) => this.handleLogout(e));

        // Dashboard refresh
        $(document).on("click", ".refresh-dashboard", (e) =>
            this.loadDashboardData()
        );

        // Statement filter
        $(document).on("click", "#filterStatement", (e) =>
            this.filterStatement(e)
        );

        // Load more transactions
        $(document).on("click", "#loadMoreTransactions", (e) =>
            this.loadMoreTransactions(e)
        );

        // Clear error when modal is opened
        $(document).on("show.bs.modal", "#loginModal", () => {
            this.hideLoginError();
            this.hideModalLoading();
        });
    }

    /**
     * Check authentication status on page load
     */
    async checkAuthStatus() {
        try {
            const response = await this.makeRequest("/check-auth", "GET");
            if (response.success) {
                this.state.isLoggedIn = true;
                this.state.currentUser =
                    response.data.user || response.data.member;
                this.state.apiToken = response.data.token;

                // Check if token is expired or about to expire
                if (response.data.token_expires_at) {
                    const expiresAt = new Date(response.data.token_expires_at);
                    const now = new Date();

                    if (expiresAt <= now) {
                        // Token has expired
                        console.log("Member token has expired");
                        this.handleTokenExpired();
                        return;
                    }

                    // Check if token expires in next 5 minutes (300 seconds)
                    const timeUntilExpiry = (expiresAt - now) / 1000; // in seconds
                    if (timeUntilExpiry < 300) {
                        console.log(
                            `Member token expires in ${Math.floor(
                                timeUntilExpiry
                            )} seconds`
                        );
                        this.showTokenExpiryWarning(
                            Math.floor(timeUntilExpiry)
                        );
                    }
                }

                this.updateUIForAuthenticatedUser();
            } else {
                this.state.isLoggedIn = false;
                this.state.currentUser = null;
                this.state.apiToken = null;
            }
        } catch (error) {
            console.log("User not authenticated");
            this.state.isLoggedIn = false;
            this.state.currentUser = null;
            this.state.apiToken = null;
        }
    }

    /**
     * Start monitoring auth status every 30 seconds
     */
    startAuthStatusMonitoring() {
        // Check every 30 seconds
        setInterval(() => {
            if (this.state.isLoggedIn) {
                this.checkAuthStatus();
            }
        }, 30000); // 30 seconds
    }

    /**
     * Handle token expiration
     */
    handleTokenExpired() {
        this.state.isLoggedIn = false;
        this.state.currentUser = null;
        this.state.apiToken = null;

        // Show modal notification
        this.showTokenExpiredModal();
    }

    /**
     * Show token expiry warning
     */
    showTokenExpiryWarning(secondsRemaining) {
        const minutes = Math.floor(secondsRemaining / 60);
        const message =
            minutes > 0
                ? `Your session will expire in ${minutes} minute${
                      minutes > 1 ? "s" : ""
                  }. Please save your work.`
                : `Your session will expire in less than a minute. Please save your work.`;

        // Only show warning once per session
        if (!this.state.expiryWarningShown) {
            this.showInfo(message);
            this.state.expiryWarningShown = true;
        }
    }

    /**
     * Show token expired modal
     */
    showTokenExpiredModal() {
        const modalHtml = `
            <div class="modal fade" id="tokenExpiredModal" tabindex="-1" data-bs-backdrop="static" data-bs-keyboard="false">
                <div class="modal-dialog modal-dialog-centered">
                    <div class="modal-content">
                        <div class="modal-header bg-warning">
                            <h5 class="modal-title">
                                <i class="fas fa-exclamation-triangle me-2"></i>
                                Session Expired
                            </h5>
                        </div>
                        <div class="modal-body text-center py-4">
                            <i class="fas fa-clock fa-3x text-warning mb-3"></i>
                            <p class="mb-3">Your session has expired due to inactivity.</p>
                            <p class="mb-0">Please log in again to continue.</p>
                        </div>
                        <div class="modal-footer justify-content-center">
                            <button type="button" class="btn btn-primary" id="reloginButton">
                                <i class="fas fa-sign-in-alt me-2"></i>
                                Log In Again
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        `;

        // Remove existing modal if any
        $("#tokenExpiredModal").remove();

        $("body").append(modalHtml);

        const modal = new bootstrap.Modal(
            document.getElementById("tokenExpiredModal")
        );
        modal.show();

        $("#reloginButton").on("click", () => {
            window.location.href = "/";
        });
    }

    /**
     * Handle login form submission
     */
    async handleLogin(event) {
        event.preventDefault();

        const form = event.target;
        const formData = new FormData(form);
        const data = Object.fromEntries(formData);

        // Determine login method and endpoint
        const loginMethod = data.login_method || "mobile";
        // Both login methods should use /member-login endpoint which handles both mobile OTP and member ID+password
        const endpoint = "/member-login";

        // Hide any existing errors
        this.hideLoginError();

        // Show modal loading state
        this.showModalLoading();

        try {
            const response = await this.makeRequest(endpoint, "POST", data);

            if (response.success) {
                this.state.isLoggedIn = true;
                this.state.currentUser =
                    response.data.user || response.data.member;
                this.state.apiToken = response.data.token;

                this.showModalSuccess("Login successful! Redirecting...");

                // Close modal and redirect
                setTimeout(() => {
                    $("#loginModal").modal("hide");
                    console.log("Redirecting to dashboard...");
                    window.location.href = "/membership/dashboard";
                }, 1500);
            } else {
                // Handle nested message object or string
                let errorMessage = "Incorrect login details. Please try again.";
                if (response.message) {
                    if (
                        typeof response.message === "object" &&
                        response.message.message
                    ) {
                        errorMessage = response.message.message;
                    } else if (typeof response.message === "string") {
                        errorMessage = response.message;
                    }
                }

                this.showLoginError(errorMessage);
            }
        } catch (error) {
            // Handle catch block errors (network errors, etc.)
            let errorMessage = "Incorrect login details. Please try again.";
            if (error.message && typeof error.message === "string") {
                errorMessage = error.message;
            }
            this.showLoginError(errorMessage);
        } finally {
            this.hideModalLoading();
        }
    }

    /**
     * Handle registration form submission
     */
    async handleRegistration(event) {
        event.preventDefault();

        const form = event.target;
        const formData = new FormData(form);
        const data = Object.fromEntries(formData);

        // Validate passwords match
        if (data.password !== data.password_confirmation) {
            this.showError("Passwords do not match");
            return;
        }

        this.showLoading("Creating account...");

        try {
            const response = await this.makeRequest("/register", "POST", data);

            if (response.success) {
                this.showSuccess(
                    response.message || "Registration successful! Redirecting to your dashboard..."
                );
                form.reset();
                $("#registerModal").modal("hide");
                
                // Redirect to dashboard after successful registration
                setTimeout(() => {
                    window.location.href = response.redirect || '/membership/dashboard';
                }, 1500);
            } else {
                this.showError(response.message || "Registration failed");
            }
        } catch (error) {
            this.showError("Registration failed. Please try again.");
        } finally {
            this.hideLoading();
        }
    }

    /**
     * Handle forgot password form submission
     */
    async handleForgotPassword(event) {
        event.preventDefault();

        const form = event.target;
        const formData = new FormData(form);
        const data = Object.fromEntries(formData);

        this.showLoading("Sending reset link...");

        try {
            const response = await this.makeRequest(
                "/forgot-password",
                "POST",
                data
            );

            if (response.success) {
                this.showSuccess("Password reset link sent to your email!");
                form.reset();
                $("#forgotPasswordModal").modal("hide");
            } else {
                this.showError(response.message || "Failed to send reset link");
            }
        } catch (error) {
            this.showError("Failed to send reset link. Please try again.");
        } finally {
            this.hideLoading();
        }
    }

    /**
     * Handle profile update
     */
    async handleProfileUpdate(event) {
        event.preventDefault();

        const form = event.target;
        const formData = new FormData(form);
        const data = Object.fromEntries(formData);

        this.showLoading("Updating profile...");

        try {
            const response = await this.makeRequest(
                "/update-profile",
                "POST",
                data
            );

            if (response.success) {
                this.showSuccess("Profile updated successfully!");
                this.state.currentUser = response.data.user;
                this.updateUserDisplay();
            } else {
                this.showError(response.message || "Failed to update profile");
            }
        } catch (error) {
            this.showError("Failed to update profile. Please try again.");
        } finally {
            this.hideLoading();
        }
    }

    /**
     * Handle password change
     */
    async handlePasswordChange(event) {
        event.preventDefault();

        const form = event.target;
        const formData = new FormData(form);
        const data = Object.fromEntries(formData);

        this.showLoading("Updating password...");

        try {
            const response = await this.makeRequest(
                "/update-password",
                "POST",
                data
            );

            if (response.success) {
                this.showSuccess("Password updated successfully!");
                form.reset();
            } else {
                this.showError(response.message || "Failed to update password");
            }
        } catch (error) {
            this.showError("Failed to update password. Please try again.");
        } finally {
            this.hideLoading();
        }
    }

    /**
     * Handle reward redemption
     */
    async handleRewardRedemption(event) {
        event.preventDefault();

        const button = event.target;
        const rewardId = button.getAttribute("data-reward-id");
        const rewardTitle = button.getAttribute("data-reward-title");
        const pointsRequired = button.getAttribute("data-points-required");

        // Show confirmation modal
        const confirmed = await this.showConfirmation(
            "Redeem Reward",
            `Are you sure you want to redeem "${rewardTitle}" for ${pointsRequired} points?`
        );

        if (!confirmed) return;

        this.showLoading("Redeeming reward...");

        try {
            const response = await this.makeRequest("/redeem-reward", "POST", {
                reward_id: rewardId,
            });

            if (response.success) {
                this.showSuccess("Reward redeemed successfully!");

                // Update button state
                button.innerHTML = '<i class="fas fa-check me-2"></i>Redeemed!';
                button.classList.remove("btn-primary");
                button.classList.add("btn-success");
                button.disabled = true;

                // Update points display
                this.updatePointsDisplay(response.data.remaining_points);

                // Refresh rewards if we're on the rewards page
                if (window.location.pathname.includes("myrewards")) {
                    this.loadRewardsData();
                }
            } else {
                this.showError(response.message || "Failed to redeem reward");
            }
        } catch (error) {
            this.showError("Failed to redeem reward. Please try again.");
        } finally {
            this.hideLoading();
        }
    }

    /**
     * Handle logout
     */
    async handleLogout(event) {
        event.preventDefault();

        this.showLoading("Logging out...");

        try {
            await this.makeRequest("/logout", "POST");

            this.state.isLoggedIn = false;
            this.state.currentUser = null;
            this.state.apiToken = null;

            this.showSuccess("Logged out successfully!");

            setTimeout(() => {
                window.location.href = "/";
            }, 1000);
        } catch (error) {
            // Even if API call fails, redirect to home
            window.location.href = "/";
        } finally {
            this.hideLoading();
        }
    }

    /**
     * Load dashboard data
     */
    async loadDashboardData() {
        const cacheKey = "dashboard_data";
        const cached = this.getFromCache(cacheKey);

        if (cached) {
            this.updateDashboardUI(cached);
            return;
        }

        this.showLoading("Loading dashboard...");

        try {
            const response = await this.makeRequest("/dashboard-data", "GET");

            if (response.success) {
                this.setCache(cacheKey, response.data);
                this.updateDashboardUI(response.data);
            } else {
                this.showError("Failed to load dashboard data");
            }
        } catch (error) {
            this.showError("Failed to load dashboard data");
        } finally {
            this.hideLoading();
        }
    }

    /**
     * Load rewards data
     */
    async loadRewardsData() {
        const cacheKey = "rewards_data";

        this.showLoading("Loading rewards...");

        try {
            const response = await this.makeRequest("/rewards-data", "GET");

            if (response.success) {
                this.setCache(cacheKey, response.data);
                this.updateRewardsUI(response.data);
            } else {
                this.showError("Failed to load rewards data");
            }
        } catch (error) {
            this.showError("Failed to load rewards data");
        } finally {
            this.hideLoading();
        }
    }

    /**
     * Load statement data
     */
    async loadStatementData(filters = {}) {
        this.showLoading("Loading statement...");

        try {
            const response = await this.makeRequest(
                "/statement-data",
                "GET",
                filters
            );

            if (response.success) {
                this.updateStatementUI(response.data);
            } else {
                this.showError("Failed to load statement data");
            }
        } catch (error) {
            this.showError("Failed to load statement data");
        } finally {
            this.hideLoading();
        }
    }

    /**
     * Filter statement
     */
    async filterStatement(event) {
        event.preventDefault();

        const form = $("#statementFilterForm")[0];
        const formData = new FormData(form);
        const filters = Object.fromEntries(formData);

        await this.loadStatementData(filters);
    }

    /**
     * Load more transactions
     */
    async loadMoreTransactions(event) {
        event.preventDefault();

        const button = event.target;
        const page = parseInt(button.getAttribute("data-page")) || 1;
        const nextPage = page + 1;

        this.showLoading("Loading more transactions...");

        try {
            const response = await this.makeRequest(
                `/statement-data?page=${nextPage}`,
                "GET"
            );

            if (response.success && response.data.transactions.length > 0) {
                this.appendTransactions(response.data.transactions);
                button.setAttribute("data-page", nextPage);

                // Hide button if no more pages
                if (!response.data.has_more) {
                    button.style.display = "none";
                }
            } else {
                button.style.display = "none";
                this.showInfo("No more transactions to load");
            }
        } catch (error) {
            this.showError("Failed to load more transactions");
        } finally {
            this.hideLoading();
        }
    }

    /**
     * Make AJAX request with retry logic
     */
    async makeRequest(endpoint, method = "GET", data = null, headers = {}) {
        const url = this.config.baseUrl + endpoint;

        const defaultHeaders = {
            "Content-Type": "application/json",
            Accept: "application/json",
            ...headers,
        };

        if (this.state.apiToken) {
            defaultHeaders["Authorization"] = `Bearer ${this.state.apiToken}`;
        }

        const requestOptions = {
            method: method.toUpperCase(),
            headers: defaultHeaders,
            timeout: this.config.timeout,
        };

        if (
            data &&
            (method.toUpperCase() === "POST" ||
                method.toUpperCase() === "PUT" ||
                method.toUpperCase() === "PATCH")
        ) {
            requestOptions.body = JSON.stringify(data);
        }

        let lastError;

        for (let attempt = 1; attempt <= this.config.retryAttempts; attempt++) {
            try {
                const response = await fetch(url, requestOptions);
                const result = await response.json();

                if (response.ok) {
                    return result;
                } else {
                    // Handle nested message object
                    let errorMessage = `HTTP ${response.status}`;
                    if (result.message) {
                        if (
                            typeof result.message === "object" &&
                            result.message.message
                        ) {
                            errorMessage = result.message.message;
                        } else if (typeof result.message === "string") {
                            errorMessage = result.message;
                        }
                    }
                    throw new Error(errorMessage);
                }
            } catch (error) {
                lastError = error;

                if (attempt < this.config.retryAttempts) {
                    await this.delay(1000 * attempt); // Exponential backoff
                    console.log(
                        `Request failed, retrying... (${attempt}/${this.config.retryAttempts})`
                    );
                }
            }
        }

        throw lastError;
    }

    /**
     * Cache management
     */
    setCache(key, data) {
        this.config.cache.set(key, {
            data: data,
            timestamp: Date.now(),
        });
    }

    getFromCache(key) {
        const cached = this.config.cache.get(key);

        if (cached && Date.now() - cached.timestamp < this.config.cacheTTL) {
            return cached.data;
        }

        this.config.cache.delete(key);
        return null;
    }

    clearCache() {
        this.config.cache.clear();
    }

    /**
     * UI Update Methods
     */
    updateUIForAuthenticatedUser() {
        $(".auth-only").show();
        $(".guest-only").hide();
        $(".user-name").text(this.state.currentUser?.name || "");
    }

    updateUserDisplay() {
        if (this.state.currentUser) {
            $(".user-name").text(this.state.currentUser.name);
            $(".user-email").text(this.state.currentUser.email);
        }
    }

    updatePointsDisplay(points) {
        $(".points-display").text(points.toLocaleString());
        $(".points-badge").text(points.toLocaleString() + " Points");
    }

    updateDashboardUI(data) {
        // Update stats cards
        $("#totalPoints").text(data.totalPoints?.toLocaleString() || "0");
        $("#pointsToNextTier").text(
            data.pointsToNextTier?.toLocaleString() || "0"
        );
        $("#availableRewards").text(data.availableRewards?.length || "0");

        // Update recent activities
        if (data.recentActivities) {
            this.renderRecentActivities(data.recentActivities);
        }

        // Update tier progress
        if (data.tierProgress) {
            this.updateTierProgress(data.tierProgress);
        }
    }

    updateRewardsUI(data) {
        // Update available rewards
        if (data.availableRewards) {
            this.renderAvailableRewards(data.availableRewards);
        }

        // Update earned rewards
        if (data.earnedRewards) {
            this.renderEarnedRewards(data.earnedRewards);
        }

        // Update points display
        if (data.userPoints !== undefined) {
            this.updatePointsDisplay(data.userPoints);
        }
    }

    updateStatementUI(data) {
        // Update summary cards
        $("#currentBalance").text(data.currentBalance?.toLocaleString() || "0");
        $("#totalEarned").text(data.totalEarned?.toLocaleString() || "0");
        $("#totalRedeemed").text(data.totalRedeemed?.toLocaleString() || "0");

        // Update transactions table
        if (data.transactions) {
            this.renderTransactions(data.transactions);
        }

        // Update monthly summary
        if (data.monthlySummary) {
            this.renderMonthlySummary(data.monthlySummary);
        }
    }

    renderRecentActivities(activities) {
        const container = $("#recentActivitiesContainer");
        if (!container.length) return;

        const html = activities
            .map(
                (activity) => `
            <div class="list-group-item border-0 py-3">
                <div class="d-flex justify-content-between align-items-start">
                    <div class="flex-grow-1">
                        <h6 class="mb-1">${activity.action}</h6>
                        <p class="mb-1 text-muted">${activity.description}</p>
                        <small class="text-muted">${activity.date}</small>
                    </div>
                    <div class="text-end">
                        <span class="badge ${
                            activity.points.startsWith("+")
                                ? "bg-success"
                                : "bg-danger"
                        } fs-6">
                            ${activity.points} pts
                        </span>
                    </div>
                </div>
            </div>
        `
            )
            .join("");

        container.html(html);
    }

    renderTransactions(transactions) {
        const tbody = $("#transactionsTableBody");
        if (!tbody.length) return;

        const html = transactions
            .map(
                (transaction) => `
            <tr>
                <td>
                    <span class="fw-semibold">${this.formatDate(
                        transaction.date
                    )}</span>
                    <br>
                    <small class="text-muted">${this.formatTime(
                        transaction.date
                    )}</small>
                </td>
                <td>
                    <div class="fw-semibold">${transaction.description}</div>
                </td>
                <td>
                    <span class="badge ${this.getTransactionBadgeClass(
                        transaction.type
                    )}">
                        <i class="${this.getTransactionIcon(
                            transaction.type
                        )} me-1"></i>
                        ${
                            transaction.type.charAt(0).toUpperCase() +
                            transaction.type.slice(1)
                        }
                    </span>
                </td>
                <td class="text-end">
                    <span class="fw-bold ${
                        transaction.points > 0 ? "text-success" : "text-danger"
                    }">
                        ${
                            transaction.points > 0 ? "+" : ""
                        }${transaction.points.toLocaleString()}
                    </span>
                </td>
                <td class="text-end">
                    <span class="fw-semibold">${transaction.balance.toLocaleString()}</span>
                </td>
                <td>
                    <code class="text-muted">${transaction.reference}</code>
                </td>
            </tr>
        `
            )
            .join("");

        tbody.html(html);
    }

    appendTransactions(transactions) {
        const tbody = $("#transactionsTableBody");
        if (!tbody.length) return;

        const html = transactions
            .map(
                (transaction) => `
            <tr>
                <td>
                    <span class="fw-semibold">${this.formatDate(
                        transaction.date
                    )}</span>
                    <br>
                    <small class="text-muted">${this.formatTime(
                        transaction.date
                    )}</small>
                </td>
                <td>
                    <div class="fw-semibold">${transaction.description}</div>
                </td>
                <td>
                    <span class="badge ${this.getTransactionBadgeClass(
                        transaction.type
                    )}">
                        <i class="${this.getTransactionIcon(
                            transaction.type
                        )} me-1"></i>
                        ${
                            transaction.type.charAt(0).toUpperCase() +
                            transaction.type.slice(1)
                        }
                    </span>
                </td>
                <td class="text-end">
                    <span class="fw-bold ${
                        transaction.points > 0 ? "text-success" : "text-danger"
                    }">
                        ${
                            transaction.points > 0 ? "+" : ""
                        }${transaction.points.toLocaleString()}
                    </span>
                </td>
                <td class="text-end">
                    <span class="fw-semibold">${transaction.balance.toLocaleString()}</span>
                </td>
                <td>
                    <code class="text-muted">${transaction.reference}</code>
                </td>
            </tr>
        `
            )
            .join("");

        tbody.append(html);
    }

    /**
     * Helper methods
     */
    formatDate(dateString) {
        const date = new Date(dateString);
        return date.toLocaleDateString("en-US", {
            month: "short",
            day: "numeric",
            year: "numeric",
        });
    }

    formatTime(dateString) {
        const date = new Date(dateString);
        return date.toLocaleTimeString("en-US", {
            hour: "numeric",
            minute: "2-digit",
            hour12: true,
        });
    }

    getTransactionBadgeClass(type) {
        const classes = {
            earned: "bg-success",
            redeemed: "bg-danger",
            bonus: "bg-warning text-dark",
        };
        return classes[type] || "bg-secondary";
    }

    getTransactionIcon(type) {
        const icons = {
            earned: "fas fa-plus",
            redeemed: "fas fa-minus",
            bonus: "fas fa-gift",
        };
        return icons[type] || "fas fa-circle";
    }

    updateTierProgress(progress) {
        const progressBar = $(".tier-progress-bar");
        if (progressBar.length) {
            const percentage = (progress.current / progress.target) * 100;
            progressBar.css("width", percentage + "%");
            progressBar.attr("aria-valuenow", progress.current);
            progressBar.attr("aria-valuemax", progress.target);
        }
    }

    /**
     * Notification methods
     */
    showLoading(message = "Loading...") {
        this.state.loading = true;

        // Show loading spinner or overlay
        if ($("#loadingOverlay").length === 0) {
            $("body").append(`
                <div id="loadingOverlay" class="position-fixed top-0 start-0 w-100 h-100 d-flex align-items-center justify-content-center" style="background: rgba(0,0,0,0.5); z-index: 9999;">
                    <div class="bg-white rounded p-4 text-center">
                        <div class="spinner-border text-primary mb-2" role="status"></div>
                        <div>${message}</div>
                    </div>
                </div>
            `);
        }
    }

    hideLoading() {
        this.state.loading = false;
        $("#loadingOverlay").remove();
    }

    showSuccess(message) {
        this.showNotification(message, "success");
    }

    showError(message) {
        this.showNotification(message, "danger");
    }

    showInfo(message) {
        this.showNotification(message, "info");
    }

    showNotification(message, type = "info") {
        const alertHtml = `
            <div class="alert alert-${type} alert-dismissible fade show position-fixed" 
                 style="top: 20px; right: 20px; z-index: 10000; min-width: 300px;" role="alert">
                ${message}
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        `;

        $("body").append(alertHtml);

        // Auto-dismiss after 5 seconds
        setTimeout(() => {
            $(".alert")
                .last()
                .fadeOut(() => {
                    $(this).remove();
                });
        }, 5000);
    }

    async showConfirmation(title, message) {
        return new Promise((resolve) => {
            const modalHtml = `
                <div class="modal fade" id="confirmationModal" tabindex="-1">
                    <div class="modal-dialog modal-dialog-centered">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title">${title}</h5>
                                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                            </div>
                            <div class="modal-body">
                                <p>${message}</p>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                <button type="button" class="btn btn-primary" id="confirmButton">Confirm</button>
                            </div>
                        </div>
                    </div>
                </div>
            `;

            $("body").append(modalHtml);

            const modal = new bootstrap.Modal(
                document.getElementById("confirmationModal")
            );
            modal.show();

            $("#confirmButton").on("click", () => {
                modal.hide();
                resolve(true);
            });

            $("#confirmationModal").on("hidden.bs.modal", () => {
                $("#confirmationModal").remove();
                resolve(false);
            });
        });
    }

    delay(ms) {
        return new Promise((resolve) => setTimeout(resolve, ms));
    }

    /**
     * Modal-specific loading and error handling methods
     */
    showModalLoading() {
        // Show overlay
        $("#loginLoadingOverlay").removeClass("d-none");

        // Update button state
        const btn = $("#loginSubmitBtn");
        btn.prop("disabled", true);
        btn.find(".login-btn-text").addClass("d-none");
        btn.find(".login-btn-loading").removeClass("d-none");
    }

    hideModalLoading() {
        // Hide overlay
        $("#loginLoadingOverlay").addClass("d-none");

        // Reset button state
        const btn = $("#loginSubmitBtn");
        btn.prop("disabled", false);
        btn.find(".login-btn-text").removeClass("d-none");
        btn.find(".login-btn-loading").addClass("d-none");
    }

    showLoginError(message) {
        const errorAlert = $("#loginErrorAlert");
        const errorMessage = $("#loginErrorMessage");

        errorMessage.text(message);
        errorAlert.removeClass("d-none");

        // Add shake animation
        errorAlert.addClass("animate__animated animate__shakeX");

        setTimeout(() => {
            errorAlert.removeClass("animate__animated animate__shakeX");
        }, 1000);
    }

    hideLoginError() {
        $("#loginErrorAlert").addClass("d-none");
    }

    showModalSuccess(message) {
        // Hide error first
        this.hideLoginError();

        // Remove any existing success alerts first
        $(".alert-success").remove();

        // Show success message in a temporary alert
        const successHtml = `
            <div class="alert alert-success mb-4" role="alert">
                <i class="fas fa-check-circle me-2"></i>
                ${message}
            </div>
        `;

        $("#loginErrorAlert").before(successHtml);

        // Remove success message after 3 seconds
        setTimeout(() => {
            $(".alert-success").fadeOut(() => {
                $(".alert-success").remove();
            });
        }, 3000);
    }
}

// Initialize the membership module when DOM is ready (only once)
$(document).ready(function () {
    // Prevent multiple initializations
    if (window.lalitMembership) {
        console.warn('LalitMembership already initialized, skipping...');
        return;
    }
    
    window.lalitMembership = new LalitMembership();
    
    // Auto-load page-specific data
    const path = window.location.pathname;

    if (path.includes("/membership/dashboard")) {
        window.lalitMembership.loadDashboardData();
    } else if (path.includes("/membership/myrewards")) {
        window.lalitMembership.loadRewardsData();
    } else if (path.includes("/membership/statement")) {
        window.lalitMembership.loadStatementData();
    }
});

