<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Donation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class DonationController extends Controller
{
  /**
   * Display a listing of the resource.
   */
  public function index()
  {
    $donations = Donation::ordered()->get();
    return view('admin.donations.index', compact('donations'));
  }

  /**
   * Show the form for creating a new resource.
   */
  public function create()
  {
    return view('admin.donations.create');
  }

  /**
   * Store a newly created resource in storage.
   */
  public function store(Request $request)
  {
    $validated = $request->validate([
      'title' => 'required|string|max:255',
      'description' => 'nullable|string',
      'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
      'donate_points' => 'required|integer|min:1',
      'donate_points_inr_value' => 'required|numeric|min:0.01',
      'donation_link' => 'nullable|url',
      'is_active' => 'boolean'
    ]);

    // Handle image upload
    if ($request->hasFile('image')) {
      $validated['image'] = $request->file('image')->store('donations', 'public');
    }

    $validated['is_active'] = $request->has('is_active');

    Donation::create($validated);

    return redirect()->route('admin.donations.index')
      ->with('success', 'Donation created successfully.');
  }

  /**
   * Display the specified resource.
   */
  public function show(Donation $donation)
  {
    return view('admin.donations.show', compact('donation'));
  }

  /**
   * Show the form for editing the specified resource.
   */
  public function edit(Donation $donation)
  {
    return view('admin.donations.edit', compact('donation'));
  }

  /**
   * Update the specified resource in storage.
   */
  public function update(Request $request, Donation $donation)
  {
    $validated = $request->validate([
      'title' => 'required|string|max:255',
      'description' => 'nullable|string',
      'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
      'donate_points' => 'required|integer|min:1',
      'donate_points_inr_value' => 'required|numeric|min:0.01',
      'donation_link' => 'nullable|url',
      'is_active' => 'boolean'
    ]);

    // Handle image upload
    if ($request->hasFile('image')) {
      // Delete old image if exists
      if ($donation->image) {
        Storage::disk('public')->delete($donation->image);
      }
      $validated['image'] = $request->file('image')->store('donations', 'public');
    }

    $validated['is_active'] = $request->has('is_active');

    $donation->update($validated);

    return redirect()->route('admin.donations.index')
      ->with('success', 'Donation updated successfully.');
  }

  /**
   * Remove the specified resource from storage.
   */
  public function destroy(Donation $donation)
  {
    try {
      // Delete the image if it exists
      if ($donation->image) {
        Storage::disk('public')->delete($donation->image);
      }

      $donation->delete();

      return redirect()->route('admin.donations.index')
        ->with('success', 'Donation deleted successfully.');
    } catch (\Exception $e) {
      return redirect()->route('admin.donations.index')
        ->with('error', 'Failed to delete donation: ' . $e->getMessage());
    }
  }

  /**
   * Update the sort order of donations.
   */
  public function updateOrder(Request $request)
  {
    try {
      $donations = $request->input('donations', []);

      foreach ($donations as $donationData) {
        Donation::where('id', $donationData['id'])
          ->update(['sort_order' => $donationData['sort_order']]);
      }

      return response()->json(['success' => true, 'message' => 'Sort order updated successfully']);
    } catch (\Exception $e) {
      return response()->json(['success' => false, 'message' => 'Failed to update sort order'], 500);
    }
  }
}
