<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ManageHome;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class ManageHomeController extends Controller { /**
 * Display a listing of the resource.
 */

    public function index() {
        // Get the first (and should be only) manage home record, or create one if none exists
        $manageHome = ManageHome::first();

        if (!$manageHome) {
            // Create a default record if none exists
            $manageHome = ManageHome::create([
                'section_1' => null,
                'section_2' => null,
                'section_3' => null,
                'section_4' => null,
                'section_5' => null,
                'banner_img' => null,
                'status' => true,
            ]);
        }

        // Redirect directly to edit the single record
        return redirect()->route('admin.manage-home.edit', $manageHome);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(ManageHome $manageHome) {
        return view('admin.manage-home.edit', compact('manageHome'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, ManageHome $manageHome) {
        $validated = $request->validate([
            'section_1' => 'nullable|string',
            'section_2' => 'nullable|string',
            'section_3' => 'nullable|string',
            'section_4' => 'nullable|string',
            'section_5' => 'nullable|string',
            'section_5_image' => 'nullable|file|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'banner_img' => 'nullable|file|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'status' => 'boolean',
        ]);

        // Handle banner image upload
        if ($request->hasFile('banner_img')) {
            $file = $request->file('banner_img');

            // Additional validation to ensure it's an image
            if (!in_array($file->getClientOriginalExtension(), ['jpg', 'jpeg', 'png', 'gif', 'webp'])) {
                return back()->withErrors(['banner_img' => 'The banner image must be a valid image file (jpg, jpeg, png, gif, webp).']);
            }

            // Delete old image if exists
            if ($manageHome->banner_img) {
                Storage::disk('public')->delete($manageHome->banner_img);
            }
            $validated['banner_img'] = $file->store('banners', 'public');
        }        // Handle section 5 image upload
        if ($request->hasFile('section_5_image')) {
            $file = $request->file('section_5_image');

            // Additional validation to ensure it's an image
            if (!in_array($file->getClientOriginalExtension(), ['jpg', 'jpeg', 'png', 'gif', 'webp'])) {
                return back()->withErrors(['section_5_image' => 'The image must be a valid image file (jpg, jpeg, png, gif, webp).']);
            }

            // Delete old image if exists
            if (isset($manageHome->section_5['image']) && $manageHome->section_5['image']) {
                Storage::disk('public')->delete($manageHome->section_5['image']);
            }

            // Store the new image
            $section5Image = $file->store('sections/loyalty', 'public');
        }

        // Convert section content to array format for JSON storage
        foreach (['section_1', 'section_2', 'section_3', 'section_4', 'section_5'] as $section) {
            if (!empty($validated[$section]) || $section === 'section_5') {
                // Special handling for section_5 which might have an image
                if ($section === 'section_5') {
                    // Preserve existing image if not replacing
                    $imageValue = isset($manageHome->section_5['image']) ? $manageHome->section_5['image'] : null;

                    // Update with new image if uploaded
                    if (isset($section5Image)) {
                        $imageValue = $section5Image;
                    }

                    // Clean and process content
                    $content = !empty($validated[$section]) ? $validated[$section] : '';
                    $content = preg_replace('/<pre class="ql-syntax"[^>]*>(.*?)<\/pre>/s', '$1', $content);
                    $content = html_entity_decode($content, ENT_QUOTES, 'UTF-8');
                    $content = preg_replace('/<p><br><\/p>/', '', $content);
                    $content = preg_replace('/<p><\/p>/', '', $content);

                    $validated[$section] = [
                        'content' => $content,
                        'image' => $imageValue
                    ];
                } else {
                    // Standard processing for other sections
                    $content = $validated[$section];
                    $content = preg_replace('/<pre class="ql-syntax"[^>]*>(.*?)<\/pre>/s', '$1', $content);
                    $content = html_entity_decode($content, ENT_QUOTES, 'UTF-8');
                    $content = preg_replace('/<p><br><\/p>/', '', $content);
                    $content = preg_replace('/<p><\/p>/', '', $content);

                    $validated[$section] = ['content' => $content];
                }
            } else {
                $validated[$section] = null;
            }
        }
        $manageHome->update($validated);

        return redirect()->route('admin.manage-home.index')
                        ->with('success', 'Home page sections updated successfully.');
    }

    /**
     * Toggle the status of the specified resource.
     */
    public function toggleStatus(ManageHome $manageHome) {
        $manageHome->update(['status' => !$manageHome->status]);

        return redirect()->back()
                        ->with('success', 'Status updated successfully.');
    }
}
