<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ParticipatingHotel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class ParticipatingHotelController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $hotels = ParticipatingHotel::ordered()->get();
        return view('admin.participating-hotels.index', compact('hotels'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.participating-hotels.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'sort_order' => 'nullable|integer|min:0',
            'is_active' => 'boolean'
        ]);

        $data = $request->only(['title', 'description', 'sort_order', 'is_active']);
        $data['is_active'] = $request->has('is_active');

        if ($request->hasFile('image')) {
            $data['image'] = $request->file('image')->store('participating-hotels', 'public');
        }

        ParticipatingHotel::create($data);

        return redirect()->route('admin.participating-hotels.index')
                        ->with('success', 'Participating hotel created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(ParticipatingHotel $participatingHotel)
    {
        return view('admin.participating-hotels.show', compact('participatingHotel'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(ParticipatingHotel $participatingHotel)
    {
        return view('admin.participating-hotels.edit', compact('participatingHotel'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, ParticipatingHotel $participatingHotel)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'sort_order' => 'nullable|integer|min:0',
            'is_active' => 'boolean'
        ]);

        $data = $request->only(['title', 'description', 'sort_order', 'is_active']);
        $data['is_active'] = $request->has('is_active');

        if ($request->hasFile('image')) {
            // Delete old image if exists
            if ($participatingHotel->image) {
                Storage::disk('public')->delete($participatingHotel->image);
            }
            $data['image'] = $request->file('image')->store('participating-hotels', 'public');
        }

        $participatingHotel->update($data);

        return redirect()->route('admin.participating-hotels.index')
                        ->with('success', 'Participating hotel updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(ParticipatingHotel $participatingHotel)
    {
        // Delete image if exists
        if ($participatingHotel->image) {
            Storage::disk('public')->delete($participatingHotel->image);
        }

        $participatingHotel->delete();

        return redirect()->route('admin.participating-hotels.index')
                        ->with('success', 'Participating hotel deleted successfully.');
    }
}
