<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Point;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class PointController extends Controller
{    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $points = Point::orderBy('sort_order')
                        ->orderBy('title')
                        ->paginate(10);
        return view('admin.points.index', compact('points'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.points.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {        $validated = $request->validate([
            'title' => 'required|max:255',
            'slug' => 'nullable|max:255|unique:points',
            'content' => 'nullable',
            'excerpt' => 'nullable',
            'hotel_stays_content' => 'nullable',
            'hotel_stays_icon' => 'nullable|string|max:255',
            'hotel_stays_title' => 'nullable|string|max:255',
            'hotel_stays_link' => 'nullable|url|max:255',
            'dining_content' => 'nullable',
            'dining_icon' => 'nullable|string|max:255',
            'dining_title' => 'nullable|string|max:255',
            'dining_link' => 'nullable|url|max:255',
            'spa_content' => 'nullable',
            'spa_icon' => 'nullable|string|max:255',
            'spa_title' => 'nullable|string|max:255',
            'spa_link' => 'nullable|url|max:255',
            'points_earning_image' => 'nullable|image|max:2048',
            'points_earning_image_alt' => 'nullable|max:255',
            'status' => 'required|in:draft,published,archived',
            'sort_order' => 'nullable|integer',
            'is_featured' => 'boolean',
            'published_at' => 'nullable|date',
            'meta_title' => 'nullable|max:255',
            'meta_description' => 'nullable',
            'meta_keywords' => 'nullable',
            'meta_robots' => 'nullable|max:255',
            'canonical_url' => 'nullable|max:255',
            'schema_markup' => 'nullable',
            'featured_image' => 'nullable|image|max:2048', // 2MB max
            'featured_image_alt' => 'nullable|max:255',
            'featured_image_caption' => 'nullable|max:255',
            'banner_image' => 'nullable|image|max:2048',
            'banner_image_alt' => 'nullable|max:255',
        ]);
        
        // Handle image uploads
        if ($request->hasFile('featured_image')) {
            $validated['featured_image'] = $request->file('featured_image')->store('points/featured', 'public');
        }
        
        if ($request->hasFile('banner_image')) {
            $validated['banner_image'] = $request->file('banner_image')->store('points/banners', 'public');
        }
        
        if ($request->hasFile('points_earning_image')) {
            $validated['points_earning_image'] = $request->file('points_earning_image')->store('points/earning', 'public');
        }
        
        // Set the author
        $validated['author_id'] = Auth::id();
        $validated['updated_by'] = Auth::id();
        
        // Generate slug if not provided
        if (empty($validated['slug'])) {
            $validated['slug'] = Str::slug($validated['title']);
        }
        
        Point::create($validated);
        
        return redirect()->route('admin.points.index')
            ->with('success', 'Point created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Point $point)
    {
        return view('admin.points.show', compact('point'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Point $point)
    {
        return view('admin.points.edit', compact('point'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Point $point)
    {        $validated = $request->validate([
            'title' => 'required|max:255',
            'slug' => 'nullable|max:255|unique:points,slug,' . $point->id,
            'content' => 'nullable',
            'excerpt' => 'nullable',
            'hotel_stays_content' => 'nullable',
            'hotel_stays_icon' => 'nullable|string|max:255',
            'hotel_stays_title' => 'nullable|string|max:255',
            'hotel_stays_link' => 'nullable|url|max:255',
            'dining_content' => 'nullable',
            'dining_icon' => 'nullable|string|max:255',
            'dining_title' => 'nullable|string|max:255',
            'dining_link' => 'nullable|url|max:255',
            'spa_content' => 'nullable',
            'spa_icon' => 'nullable|string|max:255',
            'spa_title' => 'nullable|string|max:255',
            'spa_link' => 'nullable|url|max:255',
            'points_earning_image' => 'nullable|image|max:2048',
            'points_earning_image_alt' => 'nullable|max:255',
            'status' => 'required|in:draft,published,archived',
            'sort_order' => 'nullable|integer',
            'is_featured' => 'boolean',
            'published_at' => 'nullable|date',
            'meta_title' => 'nullable|max:255',
            'meta_description' => 'nullable',
            'meta_keywords' => 'nullable',
            'meta_robots' => 'nullable|max:255',
            'canonical_url' => 'nullable|max:255',
            'schema_markup' => 'nullable',
            'featured_image' => 'nullable|image|max:2048', // 2MB max
            'featured_image_alt' => 'nullable|max:255',
            'featured_image_caption' => 'nullable|max:255',
            'banner_image' => 'nullable|image|max:2048',
            'banner_image_alt' => 'nullable|max:255',
        ]);
        
        // Handle image uploads
        if ($request->hasFile('featured_image')) {
            // Delete old image if it exists
            if ($point->featured_image) {
                Storage::disk('public')->delete($point->featured_image);
            }
            $validated['featured_image'] = $request->file('featured_image')->store('points/featured', 'public');
        }
        
        if ($request->hasFile('banner_image')) {
            // Delete old image if it exists
            if ($point->banner_image) {
                Storage::disk('public')->delete($point->banner_image);
            }
            $validated['banner_image'] = $request->file('banner_image')->store('points/banners', 'public');
        }
        
        if ($request->hasFile('points_earning_image')) {
            // Delete old image if it exists
            if ($point->points_earning_image) {
                Storage::disk('public')->delete($point->points_earning_image);
            }
            $validated['points_earning_image'] = $request->file('points_earning_image')->store('points/earning', 'public');
        }
        
        // Set updater
        $validated['updated_by'] = Auth::id();
        
        // Generate slug if not provided
        if (empty($validated['slug'])) {
            $validated['slug'] = Str::slug($validated['title']);
        }
        
        $point->update($validated);
        
        return redirect()->route('admin.points.index')
            ->with('success', 'Point updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Point $point)
    {
        // Delete associated images
        if ($point->featured_image) {
            Storage::disk('public')->delete($point->featured_image);
        }
        
        if ($point->banner_image) {
            Storage::disk('public')->delete($point->banner_image);
        }
        
        if ($point->points_earning_image) {
            Storage::disk('public')->delete($point->points_earning_image);
        }
        
        $point->delete();
        
        return redirect()->route('admin.points.index')
            ->with('success', 'Point deleted successfully.');
    }
}
