<?php

namespace App\Http\Controllers\Membership;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Member;
use App\Models\MembershipApiToken;
use App\Mail\WelcomeMember;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;

class MembershipApiController extends Controller
{
  /**
   * Third-party API configuration
   */
  private $apiBaseUrl;
  private $apiKey;
  private $apiTimeout;
  private $memberKey;

  public function __construct()
  {
    $this->apiBaseUrl = env('MEMBERSHIP_API_URL');
    $this->apiKey = env('MEMBERSHIP_API_KEY');
    $this->apiTimeout = env('MEMBERSHIP_API_TIMEOUT', 30);
    $this->memberKey = env('MEMBER_KEY');
  }

  /**
   * Get or refresh API token
   */
  private function getApiToken()
  {
    // First, try to get a valid existing token
    $token = MembershipApiToken::getCurrentToken();

    if ($token && $token->isValid()) {
      $token->markAsUsed();
      return $token->access_token;
    }

    // If no valid token, request a new one
    return $this->requestNewToken();
  }

  /**
   * Request new API token from the membership service
   */
  private function requestNewToken()
  {
    try {
      $response = Http::timeout($this->apiTimeout)
        ->withHeaders([
          'Content-Type' => 'application/json',
          'x-api-key' => $this->apiKey
        ])
        ->post($this->apiBaseUrl . '/v1/login/', [
          'username' => 'LJIWebApi@thelalit.com',
          'password' => env('MEMBERSHIP_PASSWORD'),
        ]);

      //log Api response
      Log::info('Membership API Token Response', [
        'status' => $response->status(),
        'response' => $response->json()
      ]);
      if ($response->successful()) {
        $tokenData = $response->json();

        // Deactivate old tokens
        MembershipApiToken::where('is_active', true)->update(['is_active' => false]);

        // Store new token
        $tokenRecord = MembershipApiToken::createFromResponse($tokenData);

        return $tokenRecord->access_token;
      }

      Log::error('Failed to get API token', [
        'status' => $response->status(),
        'response' => $response->json()
      ]);

      return null;
    } catch (\Exception $e) {
      Log::error('API Token Request Error: ' . $e->getMessage());
      return null;
    }
  }

  /**
   * Make API request to third-party service with automatic token refresh on 401
   */
  private function makeApiRequest($endpoint, $method = 'GET', $data = [], $headers = [], $retryCount = 0)
  {
    $maxRetries = 2;

    try {
      // Get valid token for API requests
      $token = $this->getApiToken();

      if (!$token) {
        Log::error('No valid API token available for request', ['endpoint' => $endpoint]);
        return [
          'success' => false,
          'error' => 'Unable to authenticate with membership service',
          'status' => 401
        ];
      }

      $defaultHeaders = [
        'Authorization' => 'JWT ' . $token,
        'Content-Type' => 'application/json',
        'Accept' => 'application/json',
        'x-api-key' => $this->apiKey
      ];

      $headers = array_merge($defaultHeaders, $headers);
      $url = $this->apiBaseUrl . $endpoint;

      $httpClient = Http::withHeaders($headers)->timeout($this->apiTimeout);

      switch (strtoupper($method)) {
        case 'POST':
          $response = $httpClient->post($url, $data);
          break;
        case 'PUT':
          $response = $httpClient->put($url, $data);
          break;
        case 'PATCH':
          $response = $httpClient->patch($url, $data);
          break;
        case 'DELETE':
          $response = $httpClient->delete($url, $data);
          break;
        case 'GET':
        default:
          $response = $httpClient->get($url, $data);
          break;
      }

      // Get response body first
      $responseBody = $response->body();
      $responseJson = null;

      try {
        $responseJson = $response->json();
      } catch (\Exception $e) {
        Log::warning('Failed to parse JSON response', [
          'error' => $e->getMessage(),
          'body' => $responseBody
        ]);
      }

      // Check for 401 error and signature expiration
      if ($response->status() === 401 && $retryCount < $maxRetries) {
        $isTokenExpired = false;

        // Check if it's a token expiration error
        if ($responseJson) {
          // Check message field (can be string or nested object)
          if (isset($responseJson['message'])) {
            $message = is_array($responseJson['message']) ?
              ($responseJson['message']['message'] ?? json_encode($responseJson['message'])) :
              $responseJson['message'];

            if (
              stripos($message, 'expired') !== false ||
              stripos($message, 'signature') !== false ||
              stripos($message, 'authentication_failed') !== false ||
              stripos($message, 'invalid') !== false
            ) {
              $isTokenExpired = true;
            }
          }

          // Check details.error.code field
          if (
            isset($responseJson['details']['error']['code']) &&
            $responseJson['details']['error']['code'] === 'authentication_failed'
          ) {
            $isTokenExpired = true;
          }

          // Check error field
          if (isset($responseJson['error'])) {
            $error = is_string($responseJson['error']) ? $responseJson['error'] : json_encode($responseJson['error']);
            if (
              stripos($error, 'expired') !== false ||
              stripos($error, 'signature') !== false ||
              stripos($error, 'authentication') !== false
            ) {
              $isTokenExpired = true;
            }
          }

          // Check detail field (common in Django Rest Framework)
          if (isset($responseJson['detail'])) {
            $detail = is_string($responseJson['detail']) ? $responseJson['detail'] : json_encode($responseJson['detail']);
            if (
              stripos($detail, 'expired') !== false ||
              stripos($detail, 'signature') !== false ||
              stripos($detail, 'authentication') !== false
            ) {
              $isTokenExpired = true;
            }
          }
        }

        if ($isTokenExpired) {
          Log::info('API token expired, refreshing and retrying', [
            'endpoint' => $endpoint,
            'attempt' => $retryCount + 1,
            'response' => $responseJson
          ]);

          // Force refresh token by deactivating current tokens
          MembershipApiToken::where('is_active', true)->update(['is_active' => false]);

          // Get fresh token
          $newToken = $this->requestNewToken();

          if (!$newToken) {
            Log::error('Failed to refresh token during retry', ['endpoint' => $endpoint]);
            return [
              'success' => false,
              'error' => 'Failed to refresh authentication token',
              'status' => 401
            ];
          }

          // Retry the request without old headers (so new token gets picked up)
          return $this->makeApiRequest($endpoint, $method, $data, [], $retryCount + 1);
        }
      }

      if ($response->successful()) {
        // Handle case where response might be null
        if ($responseJson === null) {
          Log::error('API returned null response despite success status', [
            'url' => $url,
            'status' => $response->status(),
            'body' => $responseBody,
            'headers' => $response->headers()
          ]);
          return [
            'success' => false,
            'error' => 'API returned invalid response format',
            'status' => $response->status()
          ];
        }

        return [
          'success' => true,
          'data' => $responseJson,
          'status' => $response->status()
        ];
      }

      // Handle error responses
      $errorMessage = 'API request failed';
      if ($responseJson && isset($responseJson['message'])) {
        // Handle nested message object
        if (is_array($responseJson['message']) && isset($responseJson['message']['message'])) {
          $errorMessage = $responseJson['message']['message'];
        } else {
          $errorMessage = $responseJson['message'];
        }
      } elseif ($responseJson && isset($responseJson['error'])) {
        $errorMessage = $responseJson['error'];
      } elseif ($responseJson && isset($responseJson['detail'])) {
        $errorMessage = $responseJson['detail'];
      } elseif (!empty($responseBody)) {
        $errorMessage = $responseBody;
      }

      Log::error('API Request Failed', [
        'url' => $url,
        'status' => $response->status(),
        'response_body' => $responseBody,
        'response_json' => $responseJson,
        'error_message' => $errorMessage,
        'retry_count' => $retryCount
      ]);

      return [
        'success' => false,
        'error' => $errorMessage,
        'response_data' => $responseJson,
        'status' => $response->status()
      ];
    } catch (\Exception $e) {
      Log::error('Membership API Error: ' . $e->getMessage(), [
        'endpoint' => $endpoint,
        'method' => $method,
        'data' => $data,
        'retry_count' => $retryCount
      ]);

      return [
        'success' => false,
        'error' => 'Unable to connect to membership service',
        'status' => 500
      ];
    }
  }

  /**
   * Handle member registration via API
   */
  public function register(Request $request)
  {
    try {
      $validator = Validator::make($request->all(), [
        'first_name' => [
          'required',
          'string',
          'max:255',
          'regex:/^[a-zA-Z\s]+$/', // Only letters and spaces
          'min:2'
        ],
        'last_name' => [
          'required',
          'string',
          'max:255',
          'regex:/^[a-zA-Z\s]+$/', // Only letters and spaces
          'min:2'
        ],
        'email' => [
          'required',
          'string',
          'email:rfc,dns', // Strict email validation
          'max:255',
          'unique:members,email'
        ],
        'mobile' => [
          'required',
          'string',
          'digits:10', // Exactly 10 digits
          'regex:/^[6-9][0-9]{9}$/', // Indian mobile number format (starts with 6-9)
          'unique:members,mobile'
        ],
        'terms_accepted' => 'required|accepted', // Terms must be accepted
        'enrollment_touchpoint' => 'nullable|integer',
        'enrollment_channel' => 'nullable|string',
        'enrolling_sponsor' => 'nullable|integer',
        'enrolling_location' => 'nullable|integer',
      ], [
        // Custom error messages
        'first_name.required' => 'First name is required.',
        'first_name.regex' => 'First name should contain only letters and spaces.',
        'first_name.min' => 'First name must be at least 2 characters.',
        'last_name.required' => 'Last name is required.',
        'last_name.regex' => 'Last name should contain only letters and spaces.',
        'last_name.min' => 'Last name must be at least 2 characters.',
        'email.required' => 'Email address is required.',
        'email.email' => 'Please enter a valid email address.',
        'email.unique' => 'This email address is already registered.',
        'mobile.required' => 'Mobile number is required.',
        'mobile.digits' => 'Mobile number must be exactly 10 digits.',
        'mobile.regex' => 'Please enter a valid Indian mobile number.',
        'mobile.unique' => 'This mobile number is already registered.',
        'terms_accepted.required' => 'You must accept the terms and conditions.',
        'terms_accepted.accepted' => 'You must accept the terms and conditions.',
      ]);

      if ($validator->fails()) {
        if ($request->expectsJson() || $request->ajax()) {
          return response()->json([
            'success' => false,
            'message' => [
              'user' => $validator->errors()->toArray(),
              'scope' => 'field'
            ],
            'details' => [
              'error' => [
                'user' => $validator->errors()->toArray(),
                'scope' => 'field'
              ]
            ],
            'status_code' => 422
          ], 422);
        }
        return back()->withErrors($validator)->withInput();
      }

      // Prepare registration data according to API format
      // Note: extra_data must be an object {}, not an array []
      $registrationData = [
        'extra_data' => (object) ($request->extra_data ?? []),
        'user' => [
          'email' => $request->email,
          'first_name' => $request->first_name,
          'last_name' => $request->last_name
        ],
        'mobile' => '+91' . $request->mobile,
        'enrollment_touchpoint' => $request->enrollment_touchpoint ?? null,
        'enrollment_channel' => $request->enrollment_channel ?? 'WEBSITE',
        'enrolling_sponsor' => $request->enrolling_sponsor ?? 1,
        'enrolling_location' => $request->enrolling_location
      ];

      $response = $this->makeApiRequest('/v1/members/', 'POST', $registrationData);
      if ($response['status'] === 200) {
        // user already exists
        return response()->json([
          'success' => false,
          'message' => 'Member already exists with the provided email or mobile number. Please login instead.',
        ], 409);
      }

      // dummy response to test flow
      // $response = [
      //   'success' => true,
      //   'data' => $registrationData + ['id' => rand(1000, 9999)]
      // ];

      if ($response['success']) {
        $memberData = $response['data'];

        // check if member already exists locally by mobile or email
        $member = Member::where('email', $request->email)
          ->orWhere('mobile', $request->mobile)
          ->first();
        if ($member) {
          // Update existing member with API data
          $member->updateFromApiResponse($memberData, $request->all());

          // return error - member already exists
          if ($request->expectsJson() || $request->ajax()) {
            return response()->json([
              'success' => false,
              'message' => 'Member already exists with the provided email or mobile number. Please login instead.',
              'member_id' => $member->member_number ?? $member->id,
              'data' => [
                'member' => $member,
                'api_response' => $memberData
              ]
            ], 409);
          }
        } else {
          // Create member record from API response
          $member = Member::createFromApiResponse($memberData, $request->all());
        }

        // Send welcome email to member
        // try {
        //   Mail::to($member->email)->send(new WelcomeMember($member));
        //   Log::info('Welcome email sent to: ' . $member->email);
        // } catch (\Exception $e) {
        //   Log::error('Failed to send welcome email: ' . $e->getMessage());
        //   // Don't fail the registration if email fails
        // }

        // DO NOT auto-login - user needs to set password first

        // Return JSON response for AJAX requests
        if ($request->expectsJson() || $request->ajax()) {
          return response()->json([
            'success' => true,
            'message' => 'Account created successfully! Please check your mobile for OTP.',
            'member_id' => $member->member_number ?? $member->id,
            'data' => [
              'member' => $member,
              'api_response' => $memberData
            ]
          ]);
        }

        return redirect()->route('membership.set-new-password')->with('status', 'Registration successful! Please set your password.');
      }

      // Return JSON response for AJAX requests
      if ($request->expectsJson() || $request->ajax()) {
        return response()->json([
          'success' => false,
          'message' => $response['error'] ?? 'Registration failed',
          'details' => $response['response_data'] ?? null,
          'status_code' => $response['status'] ?? 500
        ], $response['status'] ?? 400);
      }

      return back()->withErrors([
        'email' => $response['error'] ?? 'Registration failed',
      ])->withInput();
    } catch (\Exception $e) {
      Log::error('Registration Error: ' . $e->getMessage(), [
        'exception' => $e->getMessage(),
        'trace' => $e->getTraceAsString(),
        'request_data' => $request->all()
      ]);

      if ($request->expectsJson() || $request->ajax()) {
        return response()->json([
          'success' => false,
          'message' => 'An error occurred during registration. Please try again.'
        ], 500);
      }

      return back()->withErrors([
        'email' => 'An error occurred during registration. Please try again.',
      ])->withInput();
    }
  }

  /**
   * Check if email already exists in members table
   */
  public function checkEmailExists(Request $request)
  {
    try {
      $validator = Validator::make($request->all(), [
        'email' => 'required|email:rfc,dns'
      ]);

      if ($validator->fails()) {
        return response()->json([
          'exists' => false,
          'valid' => false,
          'message' => 'Please enter a valid email address.'
        ]);
      }

      $email = $request->email;
      // Check including soft-deleted records
      $member = Member::withTrashed()->where('email', $email)->first();

      if ($member) {
        return response()->json([
          'exists' => true,
          'valid' => false,
          'message' => 'This email address is already registered.'
        ]);
      }

      return response()->json([
        'exists' => false,
        'valid' => true,
        'message' => 'Email is available.'
      ]);
    } catch (\Exception $e) {
      Log::error('Check Email Error: ' . $e->getMessage());
      return response()->json([
        'exists' => false,
        'valid' => true,
        'message' => 'Unable to verify email.'
      ]);
    }
  }

  /**
   * Check if mobile already exists in members table (with or without +91)
   */
  public function checkMobileExists(Request $request)
  {
    try {
      $validator = Validator::make($request->all(), [
        'mobile' => 'required|digits:10'
      ]);

      if ($validator->fails()) {
        return response()->json([
          'exists' => false,
          'valid' => false,
          'message' => 'Please enter a valid 10-digit mobile number.'
        ]);
      }

      $mobile = $request->mobile;
      $mobileWithPrefix = '+91' . $mobile;

      // Check including soft-deleted records, both with and without +91 prefix
      $member = Member::withTrashed()
        ->where(function ($query) use ($mobile, $mobileWithPrefix) {
          $query->where('mobile', $mobile)
            ->orWhere('mobile', $mobileWithPrefix);
        })
        ->first();

      if ($member) {
        return response()->json([
          'exists' => true,
          'valid' => false,
          'message' => 'This mobile number is already registered.'
        ]);
      }

      return response()->json([
        'exists' => false,
        'valid' => true,
        'message' => 'Mobile number is available.'
      ]);
    } catch (\Exception $e) {
      Log::error('Check Mobile Error: ' . $e->getMessage());
      return response()->json([
        'exists' => false,
        'valid' => true,
        'message' => 'Unable to verify mobile number.'
      ]);
    }
  }

  /**
   * Send OTP for mobile login
   */
  public function sendOtp(Request $request)
  {
    try {
      // Validate mobile number
      $validator = Validator::make($request->all(), [
        'mobile_number' => [
          'required',
          'string',
          'digits:10',
          'regex:/^[6-9][0-9]{9}$/' // Indian mobile number format
        ]
      ], [
        'mobile_number.required' => 'Mobile number is required.',
        'mobile_number.digits' => 'Mobile number must be exactly 10 digits.',
        'mobile_number.regex' => 'Please enter a valid Indian mobile number.'
      ]);

      if ($validator->fails()) {
        if ($request->expectsJson() || $request->ajax()) {
          return response()->json([
            'success' => false,
            'message' => [
              'user' => $validator->errors()->toArray(),
              'scope' => 'field'
            ],
            'details' => [
              'error' => [
                'user' => $validator->errors()->toArray(),
                'scope' => 'field'
              ]
            ],
            'status_code' => 422
          ], 422);
        }
        return back()->withErrors($validator)->withInput();
      }

      // Prepare OTP request data according to API specification
      $otpData = [
        'username' => '+91' . $request->mobile_number,
        'otp_type' => 'phone_otp'
      ];

      // Make OTP API request directly (without admin token)
      $response = Http::timeout($this->apiTimeout)
        ->withHeaders([
          'Content-Type' => 'application/json',
          'x-api-key' => $this->apiKey,
        ])
        ->post($this->apiBaseUrl . '/v1/members/login-otp/', $otpData);

      // Log API response for debugging
      Log::info('Send OTP API Response', [
        'status' => $response->status(),
        'response' => $response->json()
      ]);

      if ($response->successful()) {
        $responseData = $response->json();

        return response()->json([
          'success' => true,
          'message' => 'OTP sent successfully to your mobile number.',
          'data' => $responseData
        ]);
      } else {
        $errorData = $response->json();
        $errorMessage = 'Failed to send OTP. Please try again.';

        // Handle nested message object
        if (isset($errorData['message'])) {
          if (is_array($errorData['message']) || is_object($errorData['message'])) {
            $messageData = (array) $errorData['message'];
            $errorMessage = $messageData['message'] ?? $errorMessage;
          } else {
            $errorMessage = $errorData['message'];
          }
        } elseif (isset($errorData['error'])) {
          $errorMessage = $errorData['error'];
        } elseif (isset($errorData['detail'])) {
          $errorMessage = $errorData['detail'];
        }

        return response()->json([
          'success' => false,
          'message' => $errorMessage,
          'status_code' => $response->status()
        ], $response->status());
      }
    } catch (\Exception $e) {
      Log::error('OTP Send Error: ' . $e->getMessage(), [
        'mobile_number' => $request->mobile_number ?? 'unknown',
        'trace' => $e->getTraceAsString()
      ]);

      return response()->json([
        'success' => false,
        'message' => 'An error occurred while sending OTP. Please try again later.',
        'status_code' => 500
      ], 500);
    }
  }

  /**
   * Member login (supports both mobile+OTP and member ID+password)
   */
  public function login(Request $request)
  {
    try {
      // Determine login method
      $loginMethod = $request->input('login_method', 'mobile'); // Default to mobile login

      if ($loginMethod === 'mobile') {
        // Mobile + OTP login validation
        $validator = Validator::make($request->all(), [
          'mobile_number' => [
            'required',
            'string',
            'digits:10',
            'regex:/^[6-9][0-9]{9}$/' // Indian mobile number format
          ],
          'otp' => [
            'required',
            'string',
            'digits:6' // Exactly 6 digits for OTP
          ],
          'login_method' => 'required|string|in:mobile'
        ], [
          'mobile_number.required' => 'Mobile number is required.',
          'mobile_number.digits' => 'Mobile number must be exactly 10 digits.',
          'mobile_number.regex' => 'Please enter a valid Indian mobile number.',
          'otp.required' => 'OTP is required.',
          'otp.digits' => 'OTP must be exactly 6 digits.'
        ]);
      } else {
        // Member ID + Password login validation
        $validator = Validator::make($request->all(), [
          'membership_number' => [
            'required',
            'string',
            'min:3',
            'max:50'
          ],
          'password' => [
            'required',
            'string',
            'min:6'
          ],
          'login_method' => 'required|string|in:member'
        ], [
          'membership_number.required' => 'Membership number is required.',
          'membership_number.min' => 'Membership number must be at least 3 characters.',
          'membership_number.max' => 'Membership number cannot exceed 50 characters.',
          'password.required' => 'Password is required.',
          'password.min' => 'Password must be at least 6 characters.'
        ]);
      }

      if ($validator->fails()) {
        if ($request->expectsJson() || $request->ajax()) {
          return response()->json([
            'success' => false,
            'message' => [
              'user' => $validator->errors()->toArray(),
              'scope' => 'field'
            ],
            'details' => [
              'error' => [
                'user' => $validator->errors()->toArray(),
                'scope' => 'field'
              ]
            ],
            'status_code' => 422
          ], 422);
        }
        return back()->withErrors($validator)->withInput();
      }

      // Prepare login data and endpoint based on method
      if ($loginMethod === 'mobile') {
        // Use OTP validation endpoint for mobile login
        $loginData = [
          'otp' => $request->otp,
          'otp_type' => 'phone_otp',
          'username' => '+91' . $request->mobile_number,
        ];
        $loginEndpoint = '/v1/members/otp/validation/';
      } else {
        // Use regular login endpoint for member ID + password
        $loginData = [
          'username' => $request->membership_number,
          'password' => $request->password,
        ];
        $loginEndpoint = '/v1/members/login';
      }

      // Make login API request directly (without admin token)
      $response = Http::timeout($this->apiTimeout)
        ->withHeaders([
          'x-api-key' => $this->apiKey,
          'Content-Type' => 'application/json',
          'Accept' => 'application/json',
        ])
        ->post($this->apiBaseUrl . $loginEndpoint, $loginData);

      // Log API response for debugging
      Log::info('Member Login API Response', [
        'login_method' => $loginMethod,
        'endpoint' => $loginEndpoint,
        'status' => $response->status(),
        'response' => $response->json()
      ]);

      if ($response->successful() && $response->status() === 200) {
        $responseData = $response->json();

        // Check if login was successful
        if (isset($responseData['success']) && $responseData['success'] === true) {
          $memberData = $responseData['member_data'] ?? [];
          $memberNumber = $memberData['member_id'] ?? $request->membership_number;

          // Find or create member record
          $member = Member::findByMemberNumber($memberNumber);

          if (!$member) {
            // Create new member record from login response
            $userData = $memberData['user'] ?? [];
            $member = Member::create([
              'api_member_id' => $memberData['id'] ?? null,
              'member_number' => $memberNumber,
              'first_name' => $userData['first_name'] ?? 'Member',
              'last_name' => $userData['last_name'] ?? '',
              'email' => $userData['email'] ?? '',
              'mobile' => $memberData['mobile'] ?? '',
              'status' => 'active',
            ]);
          }

          // Update member with login response data
          $member->updateFromLoginResponse($responseData);

          // Store member info in session
          Session::put('member_id', $member->id);
          Session::put('member_number', $member->member_number);
          Session::put('member_api_id', $memberNumber); // This is the actual member_id for API calls
          Session::put('member_name', $member->full_name);
          Session::put('member_token', $responseData['token']);
          Session::put('member_points', $member->points);

          // Return JSON response for AJAX requests
          if ($request->expectsJson() || $request->ajax()) {
            return response()->json([
              'success' => true,
              'message' => 'Login successful! Welcome back, ' . $member->first_name . '.',
              'data' => [
                'member' => $member,
                'token' => $responseData['token'],
                'expires_in' => $responseData['expires_in'],
              ]
            ]);
          }

          return redirect()->route('home')->with('status', 'Login successful! Welcome back, ' . $member->first_name . '.');
        }

        // Login failed but API returned success status
        $errorMessage = $responseData['message'] ?? 'Invalid membership number or password.';

        if ($request->expectsJson() || $request->ajax()) {
          return response()->json([
            'success' => false,
            'message' => $errorMessage,
          ], 401);
        }

        return back()->withErrors([
          'membership_number' => $errorMessage,
        ])->withInput($request->only('membership_number'));
      }

      // Handle API error response
      $errorData = $response->json();
      $errorMessage = 'Invalid membership number or password.';

      if (isset($errorData['message'])) {
        // Handle nested message object
        if (is_array($errorData['message']) || is_object($errorData['message'])) {
          $messageData = (array) $errorData['message'];
          $errorMessage = $messageData['message'] ?? $errorMessage;
        } else {
          $errorMessage = $errorData['message'];
        }
      } elseif (isset($errorData['error'])) {
        $errorMessage = $errorData['error'];
      } elseif (isset($errorData['detail'])) {
        $errorMessage = $errorData['detail'];
      }

      Log::warning('Member Login Failed', [
        'status' => $response->status(),
        'membership_number' => $request->membership_number,
        'response' => $errorData
      ]);

      if ($request->expectsJson() || $request->ajax()) {
        return response()->json([
          'success' => false,
          'message' => $errorMessage,
        ], $response->status());
      }

      return back()->withErrors([
        'membership_number' => $errorMessage,
      ])->withInput($request->only('membership_number'));
    } catch (\Exception $e) {
      Log::error('Member Login Error: ' . $e->getMessage(), [
        'exception' => $e->getMessage(),
        'trace' => $e->getTraceAsString(),
        'request_data' => $request->only('membership_number')
      ]);

      if ($request->expectsJson() || $request->ajax()) {
        return response()->json([
          'success' => false,
          'message' => 'An error occurred during login. Please try again.'
        ], 500);
      }

      return back()->withErrors([
        'membership_number' => 'An error occurred during login. Please try again.',
      ])->withInput($request->only('membership_number'));
    }
  }

  /**
   * Check authentication status (AJAX endpoint)
   */
  public function checkAuthStatus(Request $request)
  {
    // Check session for member authentication
    $memberId = Session::get('member_id');
    $memberNumber = Session::get('member_number');
    $memberToken = Session::get('member_token');

    // Also check old session keys for backward compatibility
    $membershipToken = Session::get('membership_token');
    $membershipUser = Session::get('membership_user');
    $membershipAuthenticated = Session::get('membership_authenticated');

    // Check if member is logged in via member login
    if ($memberId) {
      $member = Member::find($memberId);

      if ($member) {
        // Check if member token is still valid
        if ($member->hasValidToken()) {
          return response()->json([
            'success' => true,
            'data' => [
              'member' => [
                'id' => $member->id,
                'member_number' => $member->member_number,
                'first_name' => $member->first_name,
                'last_name' => $member->last_name,
                'name' => $member->full_name,
                'email' => $member->email,
                'mobile' => $member->mobile,
                'tier' => $member->tier,
                'points' => $member->points,
              ],
              'token' => $member->member_token,
              'token_expires_at' => $member->token_expires_at ? $member->token_expires_at->toIso8601String() : null,
            ]
          ]);
        } else {
          // Token has expired, clear session
          Session::forget(['member_id', 'member_number', 'member_name', 'member_token']);

          return response()->json([
            'success' => false,
            'message' => 'Member token has expired. Please log in again.',
            'token_expired' => true
          ], 401);
        }
      }
    }

    // Check old session format (for backward compatibility)
    if ($membershipAuthenticated && $membershipToken && $membershipUser) {
      return response()->json([
        'success' => true,
        'data' => [
          'user' => $membershipUser,
          'token' => $membershipToken
        ]
      ]);
    }

    return response()->json([
      'success' => false,
      'message' => 'Membership authentication required'
    ], 401);
  }

  /**
   * Get offers from API
   */
  public function getOffers(Request $request)
  {
    try {
      // Check if member is logged in
      $memberId = Session::get('member_id');
      if (!$memberId) {
        return response()->json([
          'success' => false,
          'message' => 'Please log in to view offers'
        ], 401);
      }

      $member = Member::find($memberId);
      if (!$member || !$member->hasValidToken()) {
        return response()->json([
          'success' => false,
          'message' => 'Please log in to view offers'
        ], 401);
      }

      // Build query parameters
      $params = [
        'member_id' => $member->member_number,
        'page_size' => $request->get('page_size', 10),
        'status' => 'launched,update',
      ];

      // Add optional filters
      if ($request->get('offer_type')) {
        $params['offer_type'] = $request->get('offer_type');
      }
      if ($request->get('location_code')) {
        $params['location_code'] = $request->get('location_code');
      }
      if ($request->get('sponsor')) {
        $params['sponsor'] = $request->get('sponsor');
      }
      if ($request->get('category')) {
        $params['category'] = $request->get('category');
      }
      if ($request->get('sort_by')) {
        $sortBy = $request->get('sort_by');
        $params['ordering'] = $sortBy;

        // Add date filters based on sort_by
        if ($sortBy === 'upcoming') {
          // For upcoming offers, show from tomorrow onwards
          $params['min_start_date'] = date('Y-m-d', strtotime('+1 day'));
        } elseif ($sortBy === 'created_ts') {
          // For created_ts sorting, show up to today
          $params['max_start_date'] = date('Y-m-d');
        }
      }
      // Build query string
      $queryString = http_build_query($params);
      $endpoint = '/v1/offers';
      if (!empty($queryString)) {
        $endpoint .= '?' . $queryString;
      }

      // Make API request with automatic token refresh on 401
      $maxRetries = 2;
      $attempt = 0;
      $response = null;

      while ($attempt < $maxRetries) {
        // Get API token
        $token = $this->getApiToken();

        if (!$token) {
          Log::error('Failed to get API token for offers request');
          return response()->json([
            'success' => false,
            'message' => 'Unable to authenticate with membership service'
          ], 500);
        }

        // Make API request with member token
        $response = Http::timeout($this->apiTimeout)
          ->withHeaders([
            'Authorization' => 'JWT ' . $token,
            'x-api-key' => $this->apiKey,
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
          ])
          ->get($this->apiBaseUrl . $endpoint);

        Log::info('Offers API Response', [
          'attempt' => $attempt + 1,
          'status' => $response->status(),
          'endpoint' => $endpoint,
          'response' => $response->json()
        ]);

        // If successful or not a 401 error, break the loop
        if ($response->successful() || $response->status() !== 401) {
          break;
        }

        // If 401 and we have retries left, force token refresh
        if ($response->status() === 401 && $attempt < $maxRetries - 1) {
          Log::info('Received 401, refreshing API token and retrying', [
            'attempt' => $attempt + 1
          ]);

          // Force token refresh by requesting a new token
          $this->requestNewToken();
          $attempt++;
        } else {
          break;
        }
      }

      if ($response && $response->successful()) {
        $offers = $response->json();

        return response()->json([
          'success' => true,
          'data' => $offers
        ]);
      }

      return response()->json([
        'success' => false,
        'message' => 'Failed to fetch offers'
      ], $response ? $response->status() : 500);
    } catch (\Exception $e) {
      Log::error('Offers API Error: ' . $e->getMessage(), [
        'exception' => $e->getMessage(),
        'trace' => $e->getTraceAsString(),
      ]);

      return response()->json([
        'success' => false,
        'message' => 'Unable to fetch offers'
      ], 500);
    }
  }

  /**
   * Get single offer details from API
   */
  public function getOfferDetails(Request $request, $offerId)
  {
    try {
      // Check if member is logged in
      $memberId = Session::get('member_id');
      if (!$memberId) {
        return response()->json([
          'success' => false,
          'message' => 'Please log in to view offer details'
        ], 401);
      }

      $member = Member::find($memberId);
      if (!$member || !$member->hasValidToken()) {
        return response()->json([
          'success' => false,
          'message' => 'Please log in to view offer details'
        ], 401);
      }

      $endpoint = "/v1/offers/{$offerId}/";

      // Make API request with member token
      $token = $this->getApiToken();
      $response = Http::timeout($this->apiTimeout)
        ->withHeaders([
          // 'Authorization' => 'JWT ' . $member->member_token,
          'Authorization' => 'JWT ' . $token,
          'x-api-key' => $this->apiKey,
          'Content-Type' => 'application/json',
          'Accept' => 'application/json',
        ])
        ->get($this->apiBaseUrl . $endpoint);

      Log::info('Offer Details API Response', [
        'status' => $response->status(),
        'offer_id' => $offerId,
        'response' => $response->json()
      ]);

      if ($response->successful()) {
        $offer = $response->json();

        return response()->json([
          'success' => true,
          'data' => $offer
        ]);
      }

      return response()->json([
        'success' => false,
        'message' => 'Offer not found'
      ], $response->status());
    } catch (\Exception $e) {
      Log::error('Offer Details API Error: ' . $e->getMessage(), [
        'exception' => $e->getMessage(),
        'trace' => $e->getTraceAsString(),
        'offer_id' => $offerId
      ]);

      return response()->json([
        'success' => false,
        'message' => 'Unable to fetch offer details'
      ], 500);
    }
  }

  /**
   * Get sponsors list from API
   */
  public function getSponsors(Request $request)
  {
    try {
      // Make API request with automatic token refresh on 401
      $maxRetries = 2;
      $attempt = 0;
      $response = null;

      while ($attempt < $maxRetries) {
        // Get API token
        $token = $this->getApiToken();

        if (!$token) {
          Log::error('Failed to get API token for sponsors request');
          return response()->json([
            'success' => false,
            'message' => 'Unable to authenticate with membership service'
          ], 500);
        }

        // Make API request
        $response = Http::timeout($this->apiTimeout)
          ->withHeaders([
            'Authorization' => 'JWT ' . $token,
            'x-api-key' => $this->apiKey,
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
          ])
          ->get($this->apiBaseUrl . '/v1/sponsors/');

        Log::info('Sponsors API Response', [
          'attempt' => $attempt + 1,
          'status' => $response->status(),
          'response' => $response->json()
        ]);

        // If successful or not a 401 error, break the loop
        if ($response->successful() || $response->status() !== 401) {
          break;
        }

        // If 401 and we have retries left, force token refresh
        if ($response->status() === 401 && $attempt < $maxRetries - 1) {
          Log::info('Received 401, refreshing API token and retrying', [
            'attempt' => $attempt + 1
          ]);

          // Force token refresh by requesting a new token
          $this->requestNewToken();
          $attempt++;
        } else {
          break;
        }
      }

      if ($response && $response->successful()) {
        $sponsors = $response->json();

        return response()->json([
          'success' => true,
          'data' => $sponsors
        ]);
      }

      return response()->json([
        'success' => false,
        'message' => 'Failed to fetch sponsors'
      ], $response ? $response->status() : 500);
    } catch (\Exception $e) {
      Log::error('Sponsors API Error: ' . $e->getMessage(), [
        'exception' => $e->getMessage(),
        'trace' => $e->getTraceAsString(),
      ]);

      return response()->json([
        'success' => false,
        'message' => 'Unable to fetch sponsors'
      ], 500);
    }
  }

  /**
   * Accept or reject an offer
   */
  public function offerAcceptance(Request $request, $offerId)
  {
    try {
      // Check if member is logged in
      $memberId = Session::get('member_id');
      if (!$memberId) {
        return response()->json([
          'success' => false,
          'message' => 'Please log in to accept or reject offers'
        ], 401);
      }

      $member = Member::find($memberId);
      if (!$member || !$member->hasValidToken()) {
        return response()->json([
          'success' => false,
          'message' => 'Please log in to accept or reject offers'
        ], 401);
      }

      // Validate the accepted field
      $request->validate([
        'accepted' => 'required|boolean'
      ]);

      $accepted = $request->input('accepted');

      // Build endpoint with member_id parameter
      $endpoint = "/v1/offers/{$offerId}/acceptances/?member_id={$member->member_number}";

      // Make API request with member token
      $token = $this->getApiToken();
      $response = Http::timeout($this->apiTimeout)
        ->withHeaders([
          'Authorization' => 'JWT ' . $token,
          'x-api-key' => $this->apiKey,
          'Content-Type' => 'application/json',
          'Accept' => 'application/json',
        ])
        ->post($this->apiBaseUrl . $endpoint, [
          'accepted' => $accepted
        ]);

      Log::info('Offer Acceptance API Response', [
        'status' => $response->status(),
        'offer_id' => $offerId,
        'member_id' => $member->member_number,
        'accepted' => $accepted,
        'response' => $response->json()
      ]);

      if ($response->successful()) {
        $result = $response->json();

        return response()->json([
          'success' => true,
          'message' => $accepted ? 'Offer accepted successfully' : 'Offer rejected successfully',
          'data' => $result
        ]);
      }

      return response()->json([
        'success' => false,
        'message' => 'Failed to process offer acceptance'
      ], $response->status());
    } catch (\Exception $e) {
      Log::error('Offer Acceptance API Error: ' . $e->getMessage(), [
        'exception' => $e->getMessage(),
        'trace' => $e->getTraceAsString(),
        'offer_id' => $offerId
      ]);

      return response()->json([
        'success' => false,
        'message' => 'Unable to process offer acceptance'
      ], 500);
    }
  }

  /**
   * Handle member logout via API
   */
  public function logout(Request $request)
  {
    try {
      $memberId = Session::get('member_id');
      $memberToken = Session::get('member_token');

      // Get member token for API logout call
      $logoutToken = $memberToken;
      if (!$logoutToken && $memberId) {
        $member = Member::find($memberId);
        if ($member) {
          $logoutToken = $member->member_token;
        }
      }

      // Call API logout if we have a token
      if ($logoutToken) {
        try {
          $response = Http::timeout($this->apiTimeout)
            ->withHeaders([
              'Content-Type' => 'application/json',
              'Authorization' => 'JWT ' . $logoutToken,
              'x-api-key' => $this->memberKey,
            ])
            ->post($this->apiBaseUrl . '/v1/logout/', []);

          Log::info('Member Logout API Response', [
            'status' => $response->status(),
            'response' => $response->json()
          ]);
        } catch (\Exception $e) {
          Log::warning('Logout API call failed, but continuing with local logout', [
            'error' => $e->getMessage()
          ]);
        }
      }

      // Clear all session data
      Session::forget([
        'member_id',
        'member_number',
        'member_name',
        'member_token',
        'membership_token',
        'membership_user',
        'membership_authenticated'
      ]);

      // Return JSON response for AJAX requests
      if ($request->expectsJson() || $request->ajax()) {
        return response()->json([
          'success' => true,
          'message' => 'Logged out successfully'
        ]);
      }

      return redirect()->route('home')->with('status', 'You have been logged out successfully.');

    } catch (\Exception $e) {
      Log::error('Logout Error: ' . $e->getMessage(), [
        'exception' => $e->getMessage(),
        'trace' => $e->getTraceAsString(),
      ]);

      // Clear session even if API call failed
      Session::forget([
        'member_id',
        'member_number',
        'member_name',
        'member_token',
        'membership_token',
        'membership_user',
        'membership_authenticated'
      ]);

      if ($request->expectsJson() || $request->ajax()) {
        return response()->json([
          'success' => true,
          'message' => 'Logged out successfully'
        ]);
      }

      return redirect()->route('home')->with('status', 'You have been logged out successfully.');
    }
  }

  /**
   * Display offers page
   */
  public function offers()
  {
    return view('membership.offers');
  }

  /**
   * Display offer details page
   */
  public function offerDetails($offerId)
  {
    return view('membership.offer-details', compact('offerId'));
  }

  /**
   * Update member data via external API
   */
  public function updateMemberViaAPI($member, $request)
  {
    try {
      // Get enrolling_sponsor from member's extra_data or default to 1
      $extraData = is_string($member->extra_data) ? json_decode($member->extra_data, true) : (array) ($member->extra_data ?? []);
      $enrollingSponsor = $extraData['enrolling_sponsor'] ?? 1;

      // Prepare the API payload using the exact required structure
      $payload = [
        'user' => [
          'first_name' => $request->input('first_name', ''),
          'last_name' => $request->input('last_name', ''),
          'email' => $request->input('email', '')
        ],
        'salutation' => strtoupper($request->input('title', null)),
        'middle_name' => $request->input('middle_name', null),
        'nationality' => $request->input('nationality', null),
        'wedding_anniversary' => $request->input('marriage_anniversary', null),
        'company' => $request->input('company', null),
        // 'job_title' => $request->input('designation', null),
        'date_of_birth' => $request->input('date_of_birth', null),
        'gender' => $request->input('gender', null),
        'address_line1' => $request->input('address_line_1', null),
        'address_line2' => $request->input('address_line_2', null),
        'country' => 257, // India
        'country_name' => 'India',
        'region' => (int) $request->input('region_id', null),
        'region_name' => $request->input('region_name', null),
        'city' => (int) $request->input('city', null),
        'city_name' => $request->input('city_name', null),
        'postal_code' => $request->input('postal_code', null),
        'enrolling_sponsor' => $enrollingSponsor
      ];

      // API endpoint
      $apiUrl = $this->apiBaseUrl . "/v1/members/update/{$member->member_number}";

      Log::info('Making API call to update member', [
        'member_number' => $member->member_number,
        'api_url' => $apiUrl,
        'payload' => $payload
      ]);

      // Make the PUT request to the external API
      $response = Http::withHeaders([
        'Authorization' => 'JWT ' . $this->getApiToken(),
        'x-api-key' => $this->apiKey,
        'Content-Type' => 'application/json'
      ])->put($apiUrl, $payload);

      if ($response->successful()) {
        $responseData = $response->json();

        Log::info('API call successful', [
          'member_number' => $member->member_number,
          'response_status' => $response->status(),
          'response_data' => $responseData
        ]);

        return true;
      } else {
        Log::error('API call failed', [
          'member_number' => $member->member_number,
          'status' => $response->status(),
          'response' => $response->body()
        ]);

        return false;
      }

    } catch (\Exception $e) {
      Log::error('Exception during API call', [
        'member_number' => $member->member_number,
        'error' => $e->getMessage(),
        'trace' => $e->getTraceAsString()
      ]);

      return false;
    }
  }

  /**
   * Validate OTP send on Registration
   */
  public function validateRegistrationOtp(Request $request)
  {
    try {
      // Validate request
      $validator = Validator::make($request->all(), [
        'member_id' => 'required|string',
        'otp' => 'required|string|digits:6'
      ], [
        'member_id.required' => 'Member ID is required.',
        'otp.required' => 'OTP is required.',
        'otp.digits' => 'OTP must be exactly 6 digits.'
      ]);

      if ($validator->fails()) {
        if ($request->expectsJson() || $request->ajax()) {
          return response()->json([
            'success' => false,
            'message' => 'Validation errors',
            'errors' => $validator->errors()
          ], 422);
        }
        return back()->withErrors($validator)->withInput();
      }

      $memberId = $request->member_id;
      $otp = $request->otp;

      // Build the API endpoint
      $endpoint = "/v1/members/validate/{$memberId}/";

      // Get API token for authentication
      $token = $this->getApiToken();

      if (!$token) {
        Log::error('No valid API token available for validateRegistrationOtp', [
          'member_id' => $memberId
        ]);

        return response()->json([
          'success' => false,
          'message' => 'Unable to authenticate with API service'
        ], 401);
      }

      // Make API request to validate OTP
      $response = Http::timeout($this->apiTimeout)
        ->withHeaders([
          'Authorization' => 'JWT ' . $token,
          'x-api-key' => $this->apiKey,
          'Content-Type' => 'application/json',
          'Accept' => 'application/json'
        ])
        ->post($this->apiBaseUrl . $endpoint, [
          'otp' => $otp
        ]);

      // Log API response for debugging
      Log::info('Validate Registration OTP API Response', [
        'member_id' => $memberId,
        'status' => $response->status(),
        'endpoint' => $endpoint,
        'response' => $response->json()
      ]);

      if ($response->successful()) {
        $responseData = $response->json();

        // Check if validation was successful
        if (isset($responseData['success']) && $responseData['success'] === true) {
          // Find the member in local database
          $member = Member::where('member_number', $memberId)->first();

          if ($member) {
            // Update member token if provided
            if (isset($responseData['token'])) {
              $member->member_token = $responseData['token'];

              // Calculate token expiry (expires_in is in seconds)
              if (isset($responseData['expires_in'])) {
                $member->token_expires_at = now()->addSeconds((int) $responseData['expires_in']);
              }

              $member->save();

              // Store member information in session
              Session::put('member_id', $member->id);
              Session::put('member_number', $member->member_number);
              Session::put('member_api_id', $responseData['user'] ?? $member->member_number);
              Session::put('member_name', $member->first_name . ' ' . $member->last_name);
              Session::put('member_token', $responseData['token']);

              Log::info('Member OTP validated and logged in', [
                'member_id' => $member->id,
                'member_number' => $member->member_number
              ]);
            }
          }

          return response()->json([
            'success' => true,
            'message' => $responseData['status'] ?? 'OTP validated successfully.',
            'data' => [
              'member_id' => $memberId,
              'token' => $responseData['token'] ?? null,
              'expires_in' => $responseData['expires_in'] ?? null,
              'user' => $responseData['user'] ?? null
            ]
          ]);
        }

        // API returned success status but validation failed
        $errorMessage = $responseData['status'] ?? $responseData['message'] ?? 'OTP validation failed.';

        if ($request->expectsJson() || $request->ajax()) {
          return response()->json([
            'success' => false,
            'message' => $errorMessage
          ], 400);
        }

        return back()->withErrors([
          'otp' => $errorMessage
        ])->withInput();
      }

      // Handle API error response
      $errorData = $response->json();
      $errorMessage = 'Invalid OTP. Please try again.';

      if (isset($errorData['message'])) {
        if (is_array($errorData['message']) || is_object($errorData['message'])) {
          $messageData = (array) $errorData['message'];
          $errorMessage = $messageData['message'] ?? $errorMessage;
        } else {
          $errorMessage = $errorData['message'];
        }
      } elseif (isset($errorData['error'])) {
        $errorMessage = is_string($errorData['error']) ? $errorData['error'] : ($errorData['error']['message'] ?? $errorMessage);
      } elseif (isset($errorData['detail'])) {
        $errorMessage = $errorData['detail'];
      }

      Log::warning('Registration OTP Validation Failed', [
        'status' => $response->status(),
        'member_id' => $memberId,
        'response' => $errorData
      ]);

      if ($request->expectsJson() || $request->ajax()) {
        return response()->json([
          'success' => false,
          'message' => $errorMessage
        ], $response->status());
      }

      return back()->withErrors([
        'otp' => $errorMessage
      ])->withInput();

    } catch (\Exception $e) {
      Log::error('Validate Registration OTP Error: ' . $e->getMessage(), [
        'exception' => $e->getMessage(),
        'trace' => $e->getTraceAsString(),
        'request_data' => $request->except('otp')
      ]);

      if ($request->expectsJson() || $request->ajax()) {
        return response()->json([
          'success' => false,
          'message' => 'An error occurred while validating OTP. Please try again.'
        ], 500);
      }

      return back()->withErrors([
        'otp' => 'An error occurred while validating OTP. Please try again.'
      ])->withInput();
    }
  }


  /**
   * Send OTP for password reset
   */
  public function sendResetPasswordOtp(Request $request)
  {
    try {
      // Validate input
      $validator = Validator::make($request->all(), [
        'email' => 'required|email',
      ]);

      if ($validator->fails()) {
        if ($request->expectsJson() || $request->ajax()) {
          return response()->json([
            'success' => false,
            'message' => 'Validation errors',
            'errors' => $validator->errors()
          ], 422);
        }
        return back()->withErrors($validator)->withInput();
      }

      // Make API request to send OTP
      $response = Http::timeout($this->apiTimeout)
        ->withHeaders([
          'x-api-key' => $this->apiKey,
          'Content-Type' => 'application/json',
          'Accept' => 'application/json',
        ])
        ->post($this->apiBaseUrl . '/v1/accounts/members/reset-password/', [
          'email' => $request->email,
        ]);

      // Log API response for debugging
      Log::info('Reset Password OTP API Response', [
        'status' => $response->status(),
        'email' => $request->email,
        'response' => $response->json()
      ]);

      if ($response->successful()) {
        $responseData = $response->json();

        // Check if OTP was sent successfully
        if (isset($responseData['status']) && $responseData['status'] === 'success') {
          // Store email in session for next step
          Session::put('reset_password_email', $request->email);

          if ($request->expectsJson() || $request->ajax()) {
            return response()->json([
              'status' => 'success',
              'message' => 'OTP has been sent to your mobile number. Please check your inbox.',
              'data' => [
                'email' => $request->email,
              ]
            ]);
          }

          return back()->with('status', 'OTP has been sent to your mobile number.');
        }

        // API returned success status but not the expected response
        $errorMessage = 'Failed to send OTP. Please try again.';

        // Handle nested message object
        if (isset($responseData['message'])) {
          if (is_array($responseData['message']) || is_object($responseData['message'])) {
            $messageData = (array) $responseData['message'];
            $errorMessage = $messageData['message'] ?? $errorMessage;
          } else {
            $errorMessage = $responseData['message'];
          }
        }

        if ($request->expectsJson() || $request->ajax()) {
          return response()->json([
            'status' => 'error',
            'message' => $errorMessage,
          ], 400);
        }

        return back()->withErrors([
          'email' => $errorMessage,
        ])->withInput();
      }

      // Handle API error response (4xx, 5xx status codes)
      $errorData = $response->json();
      $errorMessage = 'Failed to send OTP. Please try again.';

      // Handle nested message object with proper extraction
      if (isset($errorData['message'])) {
        if (is_array($errorData['message']) || is_object($errorData['message'])) {
          $messageData = (array) $errorData['message'];
          $errorMessage = $messageData['message'] ?? $errorMessage;
        } else {
          $errorMessage = $errorData['message'];
        }
      } elseif (isset($errorData['error'])) {
        // Handle error field
        if (is_array($errorData['error']) || is_object($errorData['error'])) {
          $errorObj = (array) $errorData['error'];
          $errorMessage = $errorObj['message'] ?? $errorMessage;
        } else {
          $errorMessage = $errorData['error'];
        }
      } elseif (isset($errorData['detail'])) {
        $errorMessage = $errorData['detail'];
      }

      Log::warning('Reset Password OTP Failed', [
        'status' => $response->status(),
        'email' => $request->email,
        'response' => $errorData
      ]);

      if ($request->expectsJson() || $request->ajax()) {
        return response()->json([
          'status' => 'error',
          'success' => false,
          'message' => $errorMessage,
        ], $response->status());
      }

      return back()->withErrors([
        'email' => $errorMessage,
      ])->withInput();

    } catch (\Exception $e) {
      Log::error('Reset Password OTP Error: ' . $e->getMessage(), [
        'exception' => $e->getMessage(),
        'trace' => $e->getTraceAsString(),
        'request_data' => $request->only('email')
      ]);

      if ($request->expectsJson() || $request->ajax()) {
        return response()->json([
          'status' => 'error',
          'message' => 'An error occurred while sending OTP. Please try again.'
        ], 500);
      }

      return back()->withErrors([
        'email' => 'An error occurred while sending OTP. Please try again.',
      ])->withInput();
    }
  }

  /**
   * Reset password with OTP
   */
  public function resetPassword(Request $request)
  {
    try {
      // Validate input
      $validator = Validator::make($request->all(), [
        'email' => 'required|email',
        'reset_key' => 'required|string',
        'new_password' => 'required|string|min:8',
      ]);

      if ($validator->fails()) {
        if ($request->expectsJson() || $request->ajax()) {
          return response()->json([
            'success' => false,
            'message' => 'Validation errors',
            'errors' => $validator->errors()
          ], 422);
        }
        return back()->withErrors($validator)->withInput();
      }

      // Make API request to reset password
      $response = Http::timeout($this->apiTimeout)
        ->withHeaders([
          'x-api-key' => $this->apiKey,
          'Content-Type' => 'application/json',
          'Accept' => 'application/json',
        ])
        ->post($this->apiBaseUrl . '/v1/accounts/do-reset-password/', [
          'email' => $request->email,
          'reset_key' => $request->reset_key,
          'new_password' => $request->new_password,
        ]);

      // Log API response for debugging
      Log::info('Reset Password API Response', [
        'status' => $response->status(),
        'email' => $request->email,
        'response' => $response->json()
      ]);

      if ($response->successful()) {
        $responseData = $response->json();

        // Check if password was reset successfully
        if (isset($responseData['status']) && $responseData['status'] === 'success') {
          // Clear session
          Session::forget('reset_password_email');

          // Auto-login the member after password reset
          $member = Member::where('email', $request->email)->first();
          if ($member) {
            Session::put('member_id', $member->id);
            Session::put('member_email', $member->email);
            Session::put('member_name', $member->first_name . ' ' . $member->last_name);
            Session::put('is_member_logged_in', true);

            // Store member token if available
            if ($member->member_token) {
              Session::put('member_token', $member->member_token);
            }

            Log::info('Member auto-logged in after password reset: ' . $member->email);
          }

          if ($request->expectsJson() || $request->ajax()) {
            return response()->json([
              'success' => true,
              'message' => 'Password has been set successfully. Redirecting to dashboard...',
            ]);
          }

          return redirect()->route('membership.dashboard')->with('status', 'Password has been set successfully. Welcome to The Lalit Loyalty!');
        }

        // API returned HTTP 200 but validation failed - return original message structure
        if ($request->expectsJson() || $request->ajax()) {
          return response()->json([
            'success' => false,
            'message' => $responseData['message'] ?? 'Failed to reset password. Please try again.'
          ], 400);
        }

        return back()->withErrors([
          'reset_key' => is_string($responseData['message']) ? $responseData['message'] : 'Failed to reset password. Please try again.',
        ])->withInput($request->except('new_password'));
      }

      // Handle API error response - return original message structure
      $errorData = $response->json();

      Log::warning('Reset Password Failed', [
        'status' => $response->status(),
        'email' => $request->email,
        'response' => $errorData
      ]);
      // echo '<pre>';print_r($errorData); exit;
      if ($request->expectsJson() || $request->ajax()) {
        return response()->json([
          'success' => false,
          'message' => $errorData['error'] ?? 'Failed to reset password. Please check your OTP and try again.'
        ], $response->status());
      }

      // For non-AJAX, parse and combine messages for form errors
      $errorMessages = [];

      if (isset($errorData['message']) && is_array($errorData['message'])) {
        foreach ($errorData['message'] as $error) {
          if (is_array($error) && isset($error['message'])) {
            $message = preg_replace("/^\[\'(.+)\'\]$/", '$1', $error['message']);
            $errorMessages[] = $message;
          } elseif (is_string($error)) {
            $errorMessages[] = $error;
          }
        }
      } elseif (isset($errorData['message']) && is_string($errorData['message'])) {
        $errorMessages[] = $errorData['message'];
      }

      if (empty($errorMessages)) {
        $errorMessages[] = 'Failed to reset password. Please check your OTP and try again.';
      }

      $errorMessage = implode(' ', $errorMessages);

      return back()->withErrors([
        'new_password' => $errorMessage,
      ])->withInput($request->except('new_password'));

    } catch (\Exception $e) {
      Log::error('Reset Password Error: ' . $e->getMessage(), [
        'exception' => $e->getMessage(),
        'trace' => $e->getTraceAsString(),
        'request_data' => $request->except('new_password')
      ]);

      if ($request->expectsJson() || $request->ajax()) {
        return response()->json([
          'success' => false,
          'message' => 'An error occurred while resetting password. Please try again.'
        ], 500);
      }

      return back()->withErrors([
        'reset_key' => 'An error occurred while resetting password. Please try again.',
      ])->withInput($request->except('new_password'));
    }
  }

  /**
   * Change password for authenticated member
   */
  public function changePassword(Request $request)
  {
    try {
      // Validate input
      $validator = Validator::make($request->all(), [
        'current_password' => 'required|string',
        'password' => 'required|string|min:8|confirmed',
      ], [
        'current_password.required' => 'Current password is required.',
        'password.required' => 'New password is required.',
        'password.min' => 'New password must be at least 8 characters.',
        'password.confirmed' => 'Password confirmation does not match.',
      ]);

      if ($validator->fails()) {
        if ($request->expectsJson() || $request->ajax()) {
          return response()->json([
            'success' => false,
            'message' => 'Validation errors',
            'errors' => $validator->errors()
          ], 422);
        }
        return back()->withErrors($validator)->withInput();
      }

      // Check if member is logged in
      $memberId = Session::get('member_id');
      if (!$memberId) {
        if ($request->expectsJson() || $request->ajax()) {
          return response()->json([
            'success' => false,
            'message' => 'Please log in to change your password'
          ], 401);
        }
        return redirect()->route('home')->with('error', 'Please log in to change your password');
      }

      $member = Member::find($memberId);
      if (!$member || !$member->hasValidToken()) {
        if ($request->expectsJson() || $request->ajax()) {
          return response()->json([
            'success' => false,
            'message' => 'Please log in to change your password'
          ], 401);
        }
        return redirect()->route('home')->with('error', 'Please log in to change your password');
      }

      // Make API request to change password
      $response = Http::timeout($this->apiTimeout)
        ->withHeaders([
          'Authorization' => 'JWT ' . $member->member_token,
          'x-api-key' => $this->memberKey,
          'Content-Type' => 'application/json',
          'Accept' => 'application/json',
        ])
        ->put($this->apiBaseUrl . '/v1/accounts/members/change-password/', [
          'current_password' => $request->current_password,
          'password' => $request->password,
        ]);

      // Log API response for debugging
      Log::info('Change Password API Response', [
        'status' => $response->status(),
        'member_id' => $member->id,
        'response' => $response->json()
      ]);

      if ($response->successful()) {
        $responseData = $response->json();

        // Check if password was changed successfully
        if (isset($responseData['status']) && $responseData['status'] === 'success') {
          if ($request->expectsJson() || $request->ajax()) {
            return response()->json([
              'success' => true,
              'message' => 'Password changed successfully.',
            ]);
          }

          return back()->with('success', 'Password changed successfully.');
        }

        // API returned success status but not success
        $errorMessage = 'Failed to change password. Please try again.';

        // Handle nested message object
        if (isset($responseData['message'])) {
          if (is_array($responseData['message']) || is_object($responseData['message'])) {
            $messageData = (array) $responseData['message'];
            $errorMessage = $messageData['message'] ?? $errorMessage;
          } else {
            $errorMessage = $responseData['message'];
          }
        }

        if ($request->expectsJson() || $request->ajax()) {
          return response()->json([
            'success' => false,
            'message' => $errorMessage,
          ], 400);
        }

        return back()->withErrors([
          'current_password' => $errorMessage,
        ])->withInput($request->except(['current_password', 'password', 'password_confirmation']));
      }

      // Handle API error response
      $errorData = $response->json();
      $errorMessage = 'Failed to change password. Please check your current password.';

      // Handle nested message object
      if (isset($errorData['message'])) {
        if (is_array($errorData['message']) || is_object($errorData['message'])) {
          $messageData = (array) $errorData['message'];
          $errorMessage = $messageData['message'] ?? $errorMessage;
        } else {
          $errorMessage = $errorData['message'];
        }
      } elseif (isset($errorData['error'])) {
        $errorMessage = $errorData['error'];
      } elseif (isset($errorData['detail'])) {
        $errorMessage = $errorData['detail'];
      }

      Log::warning('Change Password Failed', [
        'status' => $response->status(),
        'member_id' => $member->id,
        'response' => $errorData
      ]);

      if ($request->expectsJson() || $request->ajax()) {
        return response()->json([
          'success' => false,
          'message' => $errorMessage,
        ], $response->status());
      }

      return back()->withErrors([
        'current_password' => $errorMessage,
      ])->withInput($request->except(['current_password', 'password', 'password_confirmation']));

    } catch (\Exception $e) {
      Log::error('Change Password Error: ' . $e->getMessage(), [
        'exception' => $e->getMessage(),
        'trace' => $e->getTraceAsString(),
        'request_data' => $request->except(['current_password', 'password', 'password_confirmation'])
      ]);

      if ($request->expectsJson() || $request->ajax()) {
        return response()->json([
          'success' => false,
          'message' => 'An error occurred while changing password. Please try again.'
        ], 500);
      }

      return back()->withErrors([
        'current_password' => 'An error occurred while changing password. Please try again.',
      ])->withInput($request->except(['current_password', 'password', 'password_confirmation']));
    }
  }

  /**
   * Get transaction history (TRIM BITS) for the logged-in member
   */
  public function getTransactionHistory(Request $request)
  {
    try {
      // Get member ID from session
      $memberId = Session::get('member_id');
      $memberApiId = Session::get('member_api_id'); // This is the actual member_id for API
      $memberToken = Session::get('member_token');

      if (!$memberId || !$memberApiId || !$memberToken) {
        return response()->json([
          'success' => false,
          'message' => 'Member not logged in'
        ], 401);
      }

      // Get admin API token for trim/bits endpoint
      $apiToken = $this->getApiToken();

      if (!$apiToken) {
        return response()->json([
          'success' => false,
          'message' => 'Unable to authenticate with API'
        ], 401);
      }

      // Make API request to get transaction history
      $response = Http::timeout($this->apiTimeout)
        ->withHeaders([
          'Authorization' => 'JWT ' . $apiToken,
          'x-api-key' => $this->apiKey,
          'Accept' => 'application/json',
        ])
        ->get($this->apiBaseUrl . '/v1/bits/', [
          'member_id' => $memberApiId,
          // 'bit_type' => 'SPEND',
          'page_size' => 500,
          'ordering' => 'created_ts'
        ]);

      Log::info('Transaction history API response', [
        'member_api_id' => $memberApiId,
        'status' => $response->status(),
        'body' => $response->body()
      ]);

      if ($response->successful()) {
        $allTransactions = $response->json();

        // Filter to show donation transactions OR transactions with REWARD_POINTS in offer_actions
        $transactions = array_filter($allTransactions, function ($transaction) {
          // Check if it's a donation
          $isDonation = isset($transaction['header']['h_bit_type_name']) &&
            $transaction['header']['h_bit_type_name'] === 'Donation';

          // Check if offer_actions contains REWARD_POINTS type
          $hasRewardPoints = false;
          if (isset($transaction['offer_actions']) && is_array($transaction['offer_actions'])) {
            foreach ($transaction['offer_actions'] as $action) {
              if (isset($action['type']) && $action['type'] === 'REWARD_POINTS') {
                $hasRewardPoints = true;
                break;
              }
            }
          }

          return $isDonation || $hasRewardPoints;
        });

        // Reindex array to avoid gaps in JSON output
        $transactions = array_values($transactions);

        return response()->json([
          'success' => true,
          'transactions' => $transactions
        ]);
      } else {
        $errorData = $response->json();
        $errorMessage = 'Failed to fetch transaction history';

        // Extract nested error message if present
        if (isset($errorData['message'])) {
          if (is_object($errorData['message']) || is_array($errorData['message'])) {
            $messageObj = (array) $errorData['message'];
            $errorMessage = $messageObj['message'] ?? $errorMessage;
          } else {
            $errorMessage = $errorData['message'];
          }
        }

        Log::error('Failed to fetch transaction history', [
          'status' => $response->status(),
          'error' => $errorMessage,
          'response' => $errorData
        ]);

        return response()->json([
          'success' => false,
          'message' => $errorMessage
        ], $response->status());
      }
    } catch (\Exception $e) {
      Log::error('Exception in getTransactionHistory', [
        'exception' => $e->getMessage(),
        'trace' => $e->getTraceAsString()
      ]);

      return response()->json([
        'success' => false,
        'message' => 'An error occurred while fetching transaction history. Please try again.'
      ], 500);
    }
  }

  /**
   * Get cities for a specific region using external API
   */
  public function getCities($regionId)
  {
    try {
      // Get API token for authentication
      $token = $this->getApiToken();

      if (!$token) {
        Log::error('No valid API token available for getCities', [
          'region_id' => $regionId
        ]);

        return response()->json([
          'success' => false,
          'message' => 'Unable to authenticate with API service'
        ], 401);
      }

      // Build the API endpoint
      $endpoint = "/v1/cities/{$regionId}/";

      // Make API call using consistent pattern
      $response = Http::timeout($this->apiTimeout)
        ->withHeaders([
          'Authorization' => 'JWT ' . $token,
          'x-api-key' => $this->apiKey,
          'Content-Type' => 'application/json',
          'Accept' => 'application/json'
        ])
        ->get($this->apiBaseUrl . $endpoint);

      // Log API response for debugging
      Log::info('Cities API Response', [
        'region_id' => $regionId,
        'status' => $response->status(),
        'endpoint' => $endpoint
      ]);

      if ($response->successful()) {
        $cities = $response->json();

        // Transform the data to match our frontend expectations
        $transformedCities = array_map(function ($city) {
          return [
            'id' => $city['id'],
            'name' => $city['name'],
            'city_code' => $city['city_code'] ?? '',
          ];
        }, $cities);

        return response()->json([
          'success' => true,
          'data' => $transformedCities
        ]);
      }

      // Handle API error response
      Log::error('Cities API call failed', [
        'region_id' => $regionId,
        'status' => $response->status(),
        'response' => $response->body()
      ]);

      return response()->json([
        'success' => false,
        'message' => 'Failed to fetch cities from API'
      ], $response->status());

    } catch (\Exception $e) {
      Log::error('Cities API Error: ' . $e->getMessage(), [
        'region_id' => $regionId,
        'exception' => $e->getMessage(),
        'trace' => $e->getTraceAsString()
      ]);

      return response()->json([
        'success' => false,
        'message' => 'Unable to fetch cities'
      ], 500);
    }
  }

  /**
   * Get member balances from external API
   */
  public function getMemberBalances($memberNumber)
  {
    try {
      // Get API token for authentication
      $token = $this->getApiToken();

      if (!$token) {
        Log::error('No valid API token available for getMemberBalances', [
          'member_number' => $memberNumber
        ]);
        return null;
      }

      // Build the API endpoint
      $endpoint = "/v1/members/{$memberNumber}/balances/";

      // Make API call
      $response = Http::timeout($this->apiTimeout)
        ->withHeaders([
          'Authorization' => 'JWT ' . $token,
          'x-api-key' => $this->apiKey,
          'Content-Type' => 'application/json',
          'Accept' => 'application/json'
        ])
        ->get($this->apiBaseUrl . $endpoint);

      // Log API response for debugging
      Log::info('Member Balances API Response', [
        'member_number' => $memberNumber,
        'status' => $response->status(),
        'endpoint' => $endpoint
      ]);

      if ($response->successful()) {
        return $response->json();
      }

      // Handle API error response
      Log::error('Member Balances API call failed', [
        'member_number' => $memberNumber,
        'status' => $response->status(),
        'response' => $response->body()
      ]);

      return null;

    } catch (\Exception $e) {
      Log::error('Member Balances API Error: ' . $e->getMessage(), [
        'member_number' => $memberNumber,
        'exception' => $e->getMessage(),
        'trace' => $e->getTraceAsString()
      ]);

      return null;
    }
  }

  /**
   * Redeem points for donation
   */
  public function redeemDonation(Request $request)
  {
    try {
      // Validate request
      $request->validate([
        'donation_id' => 'required|integer|exists:donations,id',
        'donation_points' => 'required|integer|min:1',
        'donation_title' => 'required|string'
      ]);

      // Get member info from session
      $memberId = Session::get('member_id');
      $memberApiId = Session::get('member_api_id');

      if (!$memberId || !$memberApiId) {
        return response()->json([
          'success' => false,
          'message' => 'Please login to continue with donation.'
        ], 401);
      }

      // Get current timestamp in required format
      $currentTime = now()->setTimezone('Asia/Kolkata')->format('Y-m-d\TH:i:sP');

      // Prepare donation data for API
      $donationData = [
        'h_bit_date' => $currentTime,
        'h_sponsor_id_text' => 'The LaLit Hotels',
        'h_sponsor_id' => 1,
        'h_bit_category' => 'ACCRUAL',
        'h_bit_type' => 'CHARITY',
        'h_program_id' => 18,
        'h_member_id' => $memberApiId,
        'lines' => [],
        'payment_details' => [],
        'h_pay_in_points' => $request->donation_points,
        'h_comment' => $request->donation_title
      ];

      // Make API request to redeem points
      $response = Http::timeout($this->apiTimeout)
        ->withHeaders([
          'Authorization' => 'JWT ' . $this->getApiToken(),
          'x-api-key' => $this->apiKey,
          'Content-Type' => 'application/json',
          'Accept' => 'application/json'
        ])
        ->post($this->apiBaseUrl . '/v1/bits/', $donationData);

      // Log API response for debugging
      Log::info('Donation Redemption API Response', [
        'member_id' => $memberApiId,
        'donation_id' => $request->donation_id,
        'points' => $request->donation_points,
        'status' => $response->status(),
        'response' => $response->json()
      ]);

      if ($response->successful()) {
        $responseData = $response->json();

        // Check if donation was successful based on API response structure
        // The API returns status: "SUCCESS" and points_redeemed: true on success
        if (isset($responseData['status']) && $responseData['status'] === 'SUCCESS') {
          // Get the new balance from loyalty_balances (LaLiT Points account ID is 2)
          $newBalance = null;
          $pointsRedeemed = $request->donation_points;

          if (isset($responseData['loyalty_balances'])) {
            foreach ($responseData['loyalty_balances'] as $balance) {
              if ($balance['loyalty_account_id'] == '2' && $balance['redeemed'] > 0) {
                $newBalance = $balance['new_balance'];
                $pointsRedeemed = $balance['redeemed'];
                break;
              }
            }
          }

          $successMessage = 'Thank you! Your donation of ' . $pointsRedeemed . ' points has been processed successfully.';
          if ($newBalance !== null) {
            $successMessage .= ' Your new balance is ' . $newBalance . ' points.';

            // Update session with new balance
            Session::put('member_points', $newBalance);
          }

          return response()->json([
            'success' => true,
            'message' => $successMessage,
            'data' => [
              'bit_id' => $responseData['bit_id'] ?? null,
              'new_balance' => $newBalance,
              'points_redeemed' => $pointsRedeemed,
              'processing_date' => $responseData['processing_date'] ?? null
            ]
          ]);
        }

        // Check for error in response
        if (isset($responseData['error']) && $responseData['error'] !== null) {
          $errorMessage = is_string($responseData['error'])
            ? $responseData['error']
            : ($responseData['error']['message'] ?? 'Failed to process donation.');

          return response()->json([
            'success' => false,
            'message' => $errorMessage
          ], 400);
        }

        // API returned 200/201 but status is not SUCCESS
        $errorMessage = 'Failed to process donation. Please try again.';

        return response()->json([
          'success' => false,
          'message' => $errorMessage
        ], 400);
      }

      // Handle API error response (non-2xx status codes)
      $errorData = $response->json();
      $errorMessage = 'Failed to process donation. Please try again.';

      if (isset($errorData['message'])) {
        if (is_array($errorData['message']) || is_object($errorData['message'])) {
          $messageData = (array) $errorData['message'];
          $errorMessage = $messageData['message'] ?? $errorMessage;
        } else {
          $errorMessage = $errorData['message'];
        }
      } elseif (isset($errorData['error'])) {
        $errorMessage = is_string($errorData['error']) ? $errorData['error'] : ($errorData['error']['message'] ?? $errorMessage);
      }

      Log::warning('Donation Redemption Failed', [
        'status' => $response->status(),
        'member_id' => $memberApiId,
        'response' => $errorData
      ]);

      return response()->json([
        'success' => false,
        'message' => $errorMessage
      ], $response->status());

    } catch (\Exception $e) {
      Log::error('Donation Redemption Error: ' . $e->getMessage(), [
        'exception' => $e->getMessage(),
        'trace' => $e->getTraceAsString()
      ]);

      return response()->json([
        'success' => false,
        'message' => 'An error occurred while processing your donation. Please try again.'
      ], 500);
    }
  }

  /**
   * Send OTP to email for verification
   */
  public function sendEmailOtp(Request $request)
  {
    try {
      // Validate email
      $validator = Validator::make($request->all(), [
        'email' => 'required|email'
      ]);

      if ($validator->fails()) {
        return response()->json([
          'success' => false,
          'message' => $validator->errors()->first('email')
        ], 422);
      }

      $email = $request->email;

      // Invalidate any previous OTPs for this email
      \App\Models\EmailOtp::invalidatePreviousOtps($email);

      // Generate new 6-digit OTP
      $otp = \App\Models\EmailOtp::generateOtp();

      // Store OTP in database with 10 minutes expiry
      \App\Models\EmailOtp::create([
        'email' => $email,
        'otp' => $otp,
        'is_verified' => false,
        'expires_at' => now()->addMinutes(10),
      ]);

      // Send OTP email using Laravel Mail
      try {
        Mail::to($email)->send(new \App\Mail\EmailOtp($otp, $email));

        Log::info('Email OTP sent successfully', [
          'email' => $email,
          'otp' => $otp // Remove this in production
        ]);
      } catch (\Exception $e) {
        Log::error('Failed to send OTP email: ' . $e->getMessage());
        throw new \Exception('Failed to send OTP email. Please try again.');
      }

      return response()->json([
        'success' => true,
        'message' => 'OTP sent successfully to your email address.'
      ]);

    } catch (\Exception $e) {
      Log::error('Send Email OTP Error: ' . $e->getMessage(), [
        'exception' => $e->getMessage(),
        'trace' => $e->getTraceAsString()
      ]);

      return response()->json([
        'success' => false,
        'message' => 'Failed to send OTP. Please try again.'
      ], 500);
    }
  }

  /**
   * Verify email OTP
   */
  public function verifyEmailOtp(Request $request)
  {
    try {
      // Validate input
      $validator = Validator::make($request->all(), [
        'email' => 'required|email',
        'otp' => 'required|digits:6'
      ]);

      if ($validator->fails()) {
        return response()->json([
          'success' => false,
          'message' => $validator->errors()->first()
        ], 422);
      }

      $email = $request->email;
      $otp = $request->otp;

      // Find valid OTP
      $emailOtp = \App\Models\EmailOtp::getValidOtp($email, $otp);

      if (!$emailOtp) {
        return response()->json([
          'success' => false,
          'message' => 'Invalid or expired OTP. Please request a new one.'
        ], 422);
      }

      // Check if expired
      if ($emailOtp->isExpired()) {
        return response()->json([
          'success' => false,
          'message' => 'OTP has expired. Please request a new one.'
        ], 422);
      }

      // Mark as verified
      $emailOtp->markAsVerified();

      Log::info('Email OTP verified successfully', [
        'email' => $email
      ]);

      return response()->json([
        'success' => true,
        'message' => 'Email verified successfully.'
      ]);

    } catch (\Exception $e) {
      Log::error('Verify Email OTP Error: ' . $e->getMessage(), [
        'exception' => $e->getMessage(),
        'trace' => $e->getTraceAsString()
      ]);

      return response()->json([
        'success' => false,
        'message' => 'Failed to verify OTP. Please try again.'
      ], 500);
    }
  }

  /**
   * Set password on registration after OTP validation
   */
  public function setPasswordOnRegistration(Request $request)
  {
    try {
      // Validate input
      $validator = Validator::make($request->all(), [
        'password' => 'required|string|min:8',
      ], [
        'password.required' => 'Password is required.',
        'password.min' => 'Password must be at least 8 characters.'
      ]);

      if ($validator->fails()) {
        if ($request->expectsJson() || $request->ajax()) {
          return response()->json([
            'success' => false,
            'message' => 'Validation errors',
            'errors' => $validator->errors()
          ], 422);
        }
        return back()->withErrors($validator)->withInput();
      }

      // Check if member is logged in (after OTP validation)
      $memberId = Session::get('member_id');
      $memberToken = Session::get('member_token');

      if (!$memberId || !$memberToken) {
        if ($request->expectsJson() || $request->ajax()) {
          return response()->json([
            'success' => false,
            'message' => 'Please validate OTP first'
          ], 401);
        }
        return back()->withErrors([
          'password' => 'Please validate OTP first'
        ]);
      }

      // Make API request to set password
      $response = Http::timeout($this->apiTimeout)
        ->withHeaders([
          'Authorization' => 'JWT ' . $memberToken,
          'x-api-key' => $this->memberKey,
          'Content-Type' => 'application/json',
          'Accept' => 'application/json'
        ])
        ->post($this->apiBaseUrl . '/v1/accounts/members/choose-password/', [
          'password' => $request->password
        ]);

      // Log API response for debugging
      Log::info('Set Password API Response', [
        'member_id' => $memberId,
        'status' => $response->status(),
        'response' => $response->json()
      ]);

      if ($response->successful()) {
        $responseData = $response->json();

        // Check if password was set successfully
        if (isset($responseData['status']) && $responseData['status'] === 'success') {
          if ($request->expectsJson() || $request->ajax()) {
            return response()->json([
              'success' => true,
              'message' => 'Password set successfully. Redirecting to dashboard...'
            ]);
          }

          return redirect()->route('membership.dashboard')->with('success', 'Registration completed successfully!');
        }

        // API returned success status but not success
        $errorMessage = $responseData['message'] ?? 'Failed to set password. Please try again.';

        if ($request->expectsJson() || $request->ajax()) {
          return response()->json([
            'success' => false,
            'message' => $errorMessage
          ], 400);
        }

        return back()->withErrors([
          'password' => $errorMessage
        ]);
      }

      // Handle API error response
      $errorData = $response->json();
      $errorMessage = 'Failed to set password. Please try again.';

      if (isset($errorData['message'])) {
        if (is_array($errorData['message']) || is_object($errorData['message'])) {
          $messageData = (array) $errorData['message'];
          $errorMessage = $messageData['message'] ?? $errorMessage;
        } else {
          $errorMessage = $errorData['message'];
        }
      } elseif (isset($errorData['error'])) {
        $errorMessage = is_string($errorData['error']) ? $errorData['error'] : ($errorData['error']['message'] ?? $errorMessage);
      } elseif (isset($errorData['detail'])) {
        $errorMessage = $errorData['detail'];
      }

      Log::warning('Set Password Failed', [
        'status' => $response->status(),
        'member_id' => $memberId,
        'response' => $errorData
      ]);

      if ($request->expectsJson() || $request->ajax()) {
        return response()->json([
          'success' => false,
          'message' => $errorMessage
        ], $response->status());
      }

      return back()->withErrors([
        'password' => $errorMessage
      ]);

    } catch (\Exception $e) {
      Log::error('Set Password Error: ' . $e->getMessage(), [
        'exception' => $e->getMessage(),
        'trace' => $e->getTraceAsString()
      ]);

      if ($request->expectsJson() || $request->ajax()) {
        return response()->json([
          'success' => false,
          'message' => 'An error occurred while setting password. Please try again.'
        ], 500);
      }

      return back()->withErrors([
        'password' => 'An error occurred while setting password. Please try again.'
      ]);
    }
  }

  /**
   * Test mail configuration
   */
  public function testMail(Request $request)
  {
    try {
      $testEmail = $request->input('email', 'test@example.com');

      // Test using Laravel Mail facade
      Mail::raw('This is a test email from The LaLit Loyalty application. If you receive this, your mail configuration is working correctly!', function ($message) use ($testEmail) {
        $message->to($testEmail)
          ->subject('Test Email - The LaLit Loyalty');
      });

      Log::info('Test email sent successfully', [
        'email' => $testEmail,
        'timestamp' => now()
      ]);

      return response()->json([
        'success' => true,
        'message' => 'Test email sent successfully to ' . $testEmail,
        'mail_config' => [
          'driver' => config('mail.default'),
          'from_address' => config('mail.from.address'),
          'from_name' => config('mail.from.name')
        ]
      ]);

    } catch (\Exception $e) {
      Log::error('Test Mail Error: ' . $e->getMessage(), [
        'exception' => $e->getMessage(),
        'trace' => $e->getTraceAsString()
      ]);

      return response()->json([
        'success' => false,
        'message' => 'Failed to send test email: ' . $e->getMessage()
      ], 500);
    }
  }
}

