<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Support\Str;

class Benefit extends Model
{
    use HasFactory;
    
    protected $fillable = [
        'title',
        'slug',
        'content',
        'excerpt',
        'custom_link',
        'status',
        'sort_order',
        'is_featured',
        'published_at',
        'meta_title',
        'meta_description',
        'meta_keywords',
        'meta_robots',
        'canonical_url',
        'og_data',
        'twitter_data',
        'schema_markup',
        'featured_image',
        'featured_image_alt',
        'featured_image_caption',
        'featured_image_meta',
        'gallery_images',
        'banner_image',
        'banner_image_alt',
        'author_id',
        'updated_by',
    ];

    protected $casts = [
        'published_at' => 'datetime',
        'is_featured' => 'boolean',
        'og_data' => 'array',
        'twitter_data' => 'array',
        'featured_image_meta' => 'array',
        'gallery_images' => 'array',
    ];
    
    /**
     * Generate a slug from the title
     */
    public static function boot()
    {
        parent::boot();
        
        static::creating(function ($benefit) {
            if (!$benefit->slug) {
                $benefit->slug = Str::slug($benefit->title);
            }
        });
        
        static::updating(function ($benefit) {
            if ($benefit->isDirty('title') && !$benefit->isDirty('slug')) {
                $benefit->slug = Str::slug($benefit->title);
            }
        });
    }
    
    /**
     * Get the author
     */
    public function author(): BelongsTo
    {
        return $this->belongsTo(User::class, 'author_id');
    }

    /**
     * Get the user who last updated this benefit
     */
    public function updatedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'updated_by');
    }
    
    /**
     * Scope a query to only include published benefits
     */
    public function scopePublished($query)
    {
        return $query->where('status', 'published')
                     ->where('published_at', '<=', now());
    }

    /**
     * Scope a query to only include featured benefits
     */
    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }
}
