<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class MembershipApiToken extends Model
{
    protected $fillable = [
        'token_type',
        'access_token',
        'refresh_token',
        'expires_in',
        'expires_at',
        'scope',
        'metadata',
        'is_active',
        'last_used_at',
    ];

    protected $casts = [
        'expires_at' => 'datetime',
        'last_used_at' => 'datetime',
        'metadata' => 'array',
        'is_active' => 'boolean',
    ];

    /**
     * Check if the token is expired
     */
    public function isExpired(): bool
    {
        return $this->expires_at < now();
    }

    /**
     * Check if the token is valid (active and not expired)
     */
    public function isValid(): bool
    {
        return $this->is_active && !$this->isExpired();
    }

    /**
     * Mark token as used
     */
    public function markAsUsed(): void
    {
        $this->update(['last_used_at' => now()]);
    }

    /**
     * Deactivate the token
     */
    public function deactivate(): void
    {
        $this->update(['is_active' => false]);
    }

    /**
     * Get the current valid token
     */
    public static function getCurrentToken(): ?self
    {
        return self::where('is_active', true)
            ->where('expires_at', '>', now())
            ->orderBy('created_at', 'desc')
            ->first();
    }

    /**
     * Create a new token from API response
     */
    public static function createFromResponse(array $tokenData): self
    {
        // Handle expires_in as string or int
        $expiresIn = isset($tokenData['expires_in']) 
            ? (int) $tokenData['expires_in'] 
            : 3600; // Default 1 hour
        
        // Handle both 'token' and 'access_token' field names
        $accessToken = $tokenData['token'] ?? $tokenData['access_token'] ?? null;
        
        if (!$accessToken) {
            throw new \Exception('No token found in API response');
        }
        
        return self::create([
            'token_type' => $tokenData['token_type'] ?? 'Bearer',
            'access_token' => $accessToken,
            'refresh_token' => $tokenData['refresh_token'] ?? null,
            'expires_in' => $expiresIn,
            'expires_at' => now()->addSeconds($expiresIn),
            'scope' => $tokenData['scope'] ?? null,
            'metadata' => $tokenData,
        ]);
    }

    /**
     * Clean up expired tokens
     */
    public static function cleanupExpiredTokens(): int
    {
        return self::where('expires_at', '<', now()->subDay())
            ->delete();
    }
}
