<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Charity;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class CharityController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $charities = Charity::orderBy('sort_order')
                            ->orderBy('name')
                            ->paginate(10);
        return view('admin.charities.index', compact('charities'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.charities.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|max:255',
            'slug' => 'nullable|max:255|unique:charities',
            'description' => 'nullable',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg,webp|max:2048',
            'logo_alt' => 'nullable|max:255',
            'website_url' => 'nullable|url|max:255',
            'status' => 'required|in:active,inactive',
            'sort_order' => 'nullable|integer',
            'is_featured' => 'boolean',
            'meta_title' => 'nullable|max:255',
            'meta_description' => 'nullable',
            'meta_keywords' => 'nullable',
            'meta_robots' => 'nullable|max:255',
            'canonical_url' => 'nullable|url|max:255',
        ]);

        // Generate slug if not provided
        if (empty($validated['slug'])) {
            $validated['slug'] = Str::slug($validated['name']);
        }

        // Handle logo upload
        if ($request->hasFile('logo')) {
            $logoPath = $request->file('logo')->store('charities/logos', 'public');
            $validated['logo'] = $logoPath;
        }

        // Set default values
        $validated['is_featured'] = $request->has('is_featured');

        Charity::create($validated);

        return redirect()->route('admin.charities.index')
                        ->with('success', 'Charity created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Charity $charity)
    {
        return view('admin.charities.show', compact('charity'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Charity $charity)
    {
        return view('admin.charities.edit', compact('charity'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Charity $charity)
    {
        $validated = $request->validate([
            'name' => 'required|max:255',
            'slug' => 'nullable|max:255|unique:charities,slug,' . $charity->id,
            'description' => 'nullable',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg,webp|max:2048',
            'logo_alt' => 'nullable|max:255',
            'website_url' => 'nullable|url|max:255',
            'status' => 'required|in:active,inactive',
            'sort_order' => 'nullable|integer',
            'is_featured' => 'boolean',
            'meta_title' => 'nullable|max:255',
            'meta_description' => 'nullable',
            'meta_keywords' => 'nullable',
            'meta_robots' => 'nullable|max:255',
            'canonical_url' => 'nullable|url|max:255',
        ]);

        // Generate slug if not provided
        if (empty($validated['slug'])) {
            $validated['slug'] = Str::slug($validated['name']);
        }

        // Handle logo upload
        if ($request->hasFile('logo')) {
            // Delete old logo if it exists
            if ($charity->logo && Storage::disk('public')->exists($charity->logo)) {
                Storage::disk('public')->delete($charity->logo);
            }
            
            $logoPath = $request->file('logo')->store('charities/logos', 'public');
            $validated['logo'] = $logoPath;
        }

        // Set default values
        $validated['is_featured'] = $request->has('is_featured');

        $charity->update($validated);

        return redirect()->route('admin.charities.index')
                        ->with('success', 'Charity updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Charity $charity)
    {
        // Delete logo if it exists
        if ($charity->logo && Storage::disk('public')->exists($charity->logo)) {
            Storage::disk('public')->delete($charity->logo);
        }

        $charity->delete();

        return redirect()->route('admin.charities.index')
                        ->with('success', 'Charity deleted successfully.');
    }
}
