<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\PackageDetail;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class PackageDetailController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        // Get the first (and should be only) package detail record, or create one if none exists
        $packageDetail = PackageDetail::first();

        if (!$packageDetail) {
            // Create a default record if none exists
            $packageDetail = PackageDetail::create([
                'heading' => 'Package Details',
                'excerpt' => null,
                'content' => null,
                'featured_image' => null,
                'package_image' => null,
                'link' => null,
                'status' => true,
                'membership_benefits_title' => 'About Membership Benefits',
                'membership_benefits_content' => null,
                'membership_benefits_disclaimer' => 'The program benefits are subject to change at the sole discretion of the management.',
            ]);
        }

        // Redirect directly to edit the single record
        return redirect()->route('admin.package-details.edit', $packageDetail);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        // Not needed since we only have one record
        return redirect()->route('admin.package-details.index');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Not needed since we only have one record
        return redirect()->route('admin.package-details.index');
    }

    /**
     * Display the specified resource.
     */
    public function show(PackageDetail $packageDetail)
    {
        return redirect()->route('admin.package-details.edit', $packageDetail);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(PackageDetail $packageDetail)
    {
        return view('admin.package-details.edit', compact('packageDetail'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, PackageDetail $packageDetail)
    {
        $validated = $request->validate([
            'heading' => 'required|string|max:255',
            'excerpt' => 'nullable|string',
            'content' => 'nullable|string',
            'featured_image' => 'nullable|file|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'package_image' => 'nullable|file|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'link' => 'nullable|url|max:255',
            'status' => 'boolean',
            'membership_benefits_title' => 'nullable|string|max:255',
            'membership_benefits_content' => 'nullable|string',
            'membership_benefits_disclaimer' => 'nullable|string',
        ]);

        // Handle featured image upload
        if ($request->hasFile('featured_image')) {
            $file = $request->file('featured_image');
            
            // Additional validation to ensure it's an image
            if (!in_array($file->getClientOriginalExtension(), ['jpg', 'jpeg', 'png', 'gif', 'webp'])) {
                return back()->withErrors(['featured_image' => 'The featured image must be a valid image file (jpg, jpeg, png, gif, webp).']);
            }
            
            // Delete old image if exists
            if ($packageDetail->featured_image) {
                Storage::disk('public')->delete($packageDetail->featured_image);
            }
            $validated['featured_image'] = $file->store('packages/featured', 'public');
        }

        // Handle package image upload
        if ($request->hasFile('package_image')) {
            $file = $request->file('package_image');
            
            // Additional validation to ensure it's an image
            if (!in_array($file->getClientOriginalExtension(), ['jpg', 'jpeg', 'png', 'gif', 'webp'])) {
                return back()->withErrors(['package_image' => 'The package image must be a valid image file (jpg, jpeg, png, gif, webp).']);
            }
            
            // Delete old image if exists
            if ($packageDetail->package_image) {
                Storage::disk('public')->delete($packageDetail->package_image);
            }
            $validated['package_image'] = $file->store('packages/images', 'public');
        }

        $packageDetail->update($validated);

        return redirect()->route('admin.package-details.index')
                        ->with('success', 'Package details updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(PackageDetail $packageDetail)
    {
        // Not needed since we only have one record
        return redirect()->route('admin.package-details.index');
    }

    /**
     * Toggle the status of the specified resource.
     */
    public function toggleStatus(PackageDetail $packageDetail)
    {
        $packageDetail->update(['status' => !$packageDetail->status]);
        
        return redirect()->back()
                        ->with('success', 'Package status updated successfully.');
    }
}
