<?php

namespace App\Controllers;

use App\Models\FileUploadModel;

class Files extends BaseController
{
  protected $fileUploadModel;

  public function __construct()
  {
    $this->fileUploadModel = new FileUploadModel();
  }

  public function index()
  {
    $session = session();

    // Check if user is logged in
    if (!$session->get('logged_in')) {
      return redirect()->to('/login');
    }

    $pager = \Config\Services::pager();

    // Handle search
    $search = $this->request->getGet('search');
    $files = [];

    if ($search) {
      $files = $this->fileUploadModel->searchFiles($search);
      $pager = null; // Disable pagination for search results
    } else {
      $files = $this->fileUploadModel->getFilesPaginated(10);
    }

    $data = [
      'title' => 'Files - Data Upload',
      'username' => $session->get('username'),
      'files' => $files,
      'pager' => $this->fileUploadModel->pager,
      'search' => $search,
      'totalFiles' => $this->fileUploadModel->countAll(),
      'fileStats' => $this->fileUploadModel->getFilesCountByType()
    ];

    return view('files', $data);
  }

  public function upload()
  {
    $session = session();

    // Check if user is logged in
    if (!$session->get('logged_in')) {
      return redirect()->to('/login');
    }

    if ($this->request->getMethod() === 'POST') {
      return $this->handleUpload();
    }

    $data = [
      'title' => 'Upload File',
      'username' => $session->get('username')
    ];

    return view('files', $data);
  }

  private function handleUpload()
  {
    // Log upload attempt for debugging
    log_message('info', 'File upload attempt started');
    log_message('info', 'Files received: ' . json_encode($_FILES));
    
    // Check if at least one file was uploaded
    $importerFile = $this->request->getFile('importer_file');
    $exporterFile = $this->request->getFile('exporter_file');
    $sourceFile = $this->request->getFile('source_file');
    
    $hasFiles = false;
    $validationRules = [];
    
    if ($importerFile && $importerFile->isValid()) {
      $validationRules['importer_file'] = 'uploaded[importer_file]|ext_in[importer_file,csv,xlsx,xls]';
      $hasFiles = true;
    }
    
    if ($exporterFile && $exporterFile->isValid()) {
      $validationRules['exporter_file'] = 'uploaded[exporter_file]|ext_in[exporter_file,csv,xlsx,xls]';
      $hasFiles = true;
    }
    
    if ($sourceFile && $sourceFile->isValid()) {
      $validationRules['source_file'] = 'uploaded[source_file]|ext_in[source_file,csv,xlsx,xls]';
      $hasFiles = true;
    }
    
    if (!$hasFiles) {
      return redirect()->back()->with('error', 'Please select at least one valid file to upload.');
    }
    
    if (!empty($validationRules) && !$this->validate($validationRules)) {
      log_message('error', 'Validation failed: ' . json_encode($this->validator->getErrors()));
      return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
    }

    $uploadPath = WRITEPATH . 'uploads/';

    // Handle file uploads
    $importerFile = $this->request->getFile('importer_file');
    $exporterFile = $this->request->getFile('exporter_file');
    $sourceFile = $this->request->getFile('source_file');

    $fileData = [
      'media_modification_time' => date('Y-m-d H:i:s')
    ];

    // Upload importer file
    if ($importerFile && $importerFile->isValid() && !$importerFile->hasMoved()) {
      $importerName = $importerFile->getRandomName();
      $importerFile->move($uploadPath, $importerName);
      $fileData['file_importer'] = $importerName;
      log_message('info', 'Importer file uploaded: ' . $importerName);
    }

    // Upload exporter file
    if ($exporterFile && $exporterFile->isValid() && !$exporterFile->hasMoved()) {
      $exporterName = $exporterFile->getRandomName();
      $exporterFile->move($uploadPath, $exporterName);
      $fileData['file_exporter'] = $exporterName;
      log_message('info', 'Exporter file uploaded: ' . $exporterName);
    }

    // Upload source file
    if ($sourceFile && $sourceFile->isValid() && !$sourceFile->hasMoved()) {
      $sourceName = $sourceFile->getRandomName();
      $sourceFile->move($uploadPath, $sourceName);
      $fileData['source_file'] = $sourceName;
      log_message('info', 'Source file uploaded: ' . $sourceName);
    }

    log_message('info', 'Attempting to save file data: ' . json_encode($fileData));
    
    // Check if a record already exists
    $existingRecord = $this->fileUploadModel->first();
    $uploadedFiles = [];
    
    if ($existingRecord) {
      // Update existing record - only update the columns that have new files
      $updateData = ['media_modification_time' => date('Y-m-d H:i:s')];
      
      // Delete old files before updating with new ones
      if (isset($fileData['file_importer'])) {
        if (!empty($existingRecord['file_importer']) && file_exists($uploadPath . $existingRecord['file_importer'])) {
          unlink($uploadPath . $existingRecord['file_importer']);
        }
        $updateData['file_importer'] = $fileData['file_importer'];
        $uploadedFiles[] = 'Importer File';
      }
      
      if (isset($fileData['file_exporter'])) {
        if (!empty($existingRecord['file_exporter']) && file_exists($uploadPath . $existingRecord['file_exporter'])) {
          unlink($uploadPath . $existingRecord['file_exporter']);
        }
        $updateData['file_exporter'] = $fileData['file_exporter'];
        $uploadedFiles[] = 'Exporter File';
      }
      
      if (isset($fileData['source_file'])) {
        if (!empty($existingRecord['source_file']) && file_exists($uploadPath . $existingRecord['source_file'])) {
          unlink($uploadPath . $existingRecord['source_file']);
        }
        $updateData['source_file'] = $fileData['source_file'];
        $uploadedFiles[] = 'Source File';
      }
      
      $success = $this->fileUploadModel->update($existingRecord['id'], $updateData);
      log_message('info', 'Updated existing record with ID: ' . $existingRecord['id']);
      
    } else {
      // Create new record if none exists
      if (isset($fileData['file_importer'])) $uploadedFiles[] = 'Importer File';
      if (isset($fileData['file_exporter'])) $uploadedFiles[] = 'Exporter File';
      if (isset($fileData['source_file'])) $uploadedFiles[] = 'Source File';
      
      $success = $this->fileUploadModel->insert($fileData);
      log_message('info', 'Created new file record');
    }
    
    if ($success) {
      log_message('info', 'Files saved successfully');
      
      $message = count($uploadedFiles) > 1 
        ? 'Successfully uploaded: ' . implode(', ', $uploadedFiles)
        : 'Successfully uploaded: ' . $uploadedFiles[0];
        
      return redirect()->to('/files')->with('success', $message);
    } else {
      log_message('error', 'Failed to save file data to database');
      $errors = $this->fileUploadModel->errors();
      log_message('error', 'Database errors: ' . json_encode($errors));
      return redirect()->back()->with('error', 'Failed to upload files');
    }
  }

  public function view($id)
  {
    $session = session();

    // Check if user is logged in
    if (!$session->get('logged_in')) {
      return redirect()->to('/login');
    }

    $file = $this->fileUploadModel->find($id);

    if (!$file) {
      throw new \CodeIgniter\Exceptions\PageNotFoundException('File not found');
    }

    $data = [
      'title' => 'View File Details',
      'username' => $session->get('username'),
      'file' => $file
    ];

    return view('file_view', $data);
  }

  public function update()
  {
    $session = session();

    // Check if user is logged in
    if (!$session->get('logged_in')) {
      return $this->response->setJSON([
        'sts' => false,
        'type' => 'error',
        'msg' => 'Unauthorized access'
      ]);
    }

    if ($this->request->getMethod() !== 'POST') {
      return $this->response->setJSON([
        'sts' => false,
        'type' => 'error',
        'msg' => 'Invalid request method'
      ]);
    }

    $fileType = $this->request->getPost('file_type');
    $uploadedFile = $this->request->getFile('file');

    if (!$uploadedFile || !$uploadedFile->isValid()) {
      return $this->response->setJSON([
        'sts' => false,
        'type' => 'error',
        'msg' => 'No valid file uploaded'
      ]);
    }

    // Validate file type
    $allowedTypes = ['csv', 'xlsx', 'xls'];
    if (!in_array($uploadedFile->getExtension(), $allowedTypes)) {
      return $this->response->setJSON([
        'sts' => false,
        'type' => 'error',
        'msg' => 'Invalid file type. Only CSV, XLSX, and XLS files are allowed.'
      ]);
    }

    $uploadPath = WRITEPATH . 'uploads/';

    // Get the single file record (there should only be one)
    $fileRecord = $this->fileUploadModel->first();

    if (!$fileRecord) {
      return $this->response->setJSON([
        'sts' => false,
        'type' => 'error',
        'msg' => 'No existing file record found'
      ]);
    }

    // Generate new filename and move file
    $newFileName = $uploadedFile->getRandomName();
    $uploadedFile->move($uploadPath, $newFileName);

    // Update the appropriate field based on file type
    $updateData = ['media_modification_time' => date('Y-m-d H:i:s')];

    switch ($fileType) {
      case 'importer':
        // Delete old file if exists
        if (!empty($fileRecord['file_importer']) && file_exists($uploadPath . $fileRecord['file_importer'])) {
          unlink($uploadPath . $fileRecord['file_importer']);
        }
        $updateData['file_importer'] = $newFileName;
        break;
      case 'exporter':
        // Delete old file if exists
        if (!empty($fileRecord['file_exporter']) && file_exists($uploadPath . $fileRecord['file_exporter'])) {
          unlink($uploadPath . $fileRecord['file_exporter']);
        }
        $updateData['file_exporter'] = $newFileName;
        break;
      case 'source':
        // Delete old file if exists
        if (!empty($fileRecord['source_file']) && file_exists($uploadPath . $fileRecord['source_file'])) {
          unlink($uploadPath . $fileRecord['source_file']);
        }
        $updateData['source_file'] = $newFileName;
        break;
      default:
        return $this->response->setJSON([
          'sts' => false,
          'type' => 'error',
          'msg' => 'Invalid file type specified'
        ]);
    }

    if ($this->fileUploadModel->update($fileRecord['id'], $updateData)) {
      return $this->response->setJSON([
        'sts' => true,
        'type' => 'success',
        'msg' => ucfirst($fileType) . ' file updated successfully'
      ]);
    } else {
      return $this->response->setJSON([
        'sts' => false,
        'type' => 'error',
        'msg' => 'Failed to update file record'
      ]);
    }
  }

  public function delete()
  {
    $session = session();

    // Check if user is logged in
    if (!$session->get('logged_in')) {
      return $this->response->setJSON([
        'sts' => false,
        'type' => 'error',
        'msg' => 'Unauthorized access'
      ]);
    }

    if ($this->request->getMethod() !== 'POST') {
      return $this->response->setJSON([
        'sts' => false,
        'type' => 'error',
        'msg' => 'Invalid request method'
      ]);
    }

    $json = $this->request->getJSON();
    $fileType = $json->file_type ?? null;

    if (!$fileType) {
      return $this->response->setJSON([
        'sts' => false,
        'type' => 'error',
        'msg' => 'File type not specified'
      ]);
    }

    // Get the single file record (there should only be one)
    $fileRecord = $this->fileUploadModel->first();

    if (!$fileRecord) {
      return $this->response->setJSON([
        'sts' => false,
        'type' => 'error',
        'msg' => 'No file record found'
      ]);
    }

    $uploadPath = WRITEPATH . 'uploads/';
    $updateData = ['media_modification_time' => date('Y-m-d H:i:s')];

    switch ($fileType) {
      case 'importer':
        if (!empty($fileRecord['file_importer'])) {
          if (file_exists($uploadPath . $fileRecord['file_importer'])) {
            unlink($uploadPath . $fileRecord['file_importer']);
          }
          $updateData['file_importer'] = null;
        } else {
          return $this->response->setJSON([
            'sts' => false,
            'type' => 'error',
            'msg' => 'Importer file not found'
          ]);
        }
        break;
      case 'exporter':
        if (!empty($fileRecord['file_exporter'])) {
          if (file_exists($uploadPath . $fileRecord['file_exporter'])) {
            unlink($uploadPath . $fileRecord['file_exporter']);
          }
          $updateData['file_exporter'] = null;
        } else {
          return $this->response->setJSON([
            'sts' => false,
            'type' => 'error',
            'msg' => 'Exporter file not found'
          ]);
        }
        break;
      case 'source':
        if (!empty($fileRecord['source_file'])) {
          if (file_exists($uploadPath . $fileRecord['source_file'])) {
            unlink($uploadPath . $fileRecord['source_file']);
          }
          $updateData['source_file'] = null;
        } else {
          return $this->response->setJSON([
            'sts' => false,
            'type' => 'error',
            'msg' => 'Source file not found'
          ]);
        }
        break;
      default:
        return $this->response->setJSON([
          'sts' => false,
          'type' => 'error',
          'msg' => 'Invalid file type specified'
        ]);
    }

    if ($this->fileUploadModel->update($fileRecord['id'], $updateData)) {
      return $this->response->setJSON([
        'sts' => true,
        'type' => 'success',
        'msg' => ucfirst($fileType) . ' file deleted successfully'
      ]);
    } else {
      return $this->response->setJSON([
        'sts' => false,
        'type' => 'error',
        'msg' => 'Failed to update file record'
      ]);
    }
  }
}