<?php

namespace App\Controllers;

use App\Models\FileUploadModel;

class Reports extends BaseController
{
  protected $fileUploadModel;

  public function __construct()
  {
    $this->fileUploadModel = new FileUploadModel();
  }

  public function index()
  {
    $session = session();

    // Check if user is logged in
    if (!$session->get('logged_in')) {
      return redirect()->to('/login');
    }

    // Get the single file record with CSV data
    $fileRecord = $this->fileUploadModel->first();
    $page_number = 0;

    $data = [
      'title' => 'Reports - Data Analysis',
      'username' => $session->get('username'),
      'fileRecord' => $fileRecord,
      'importerData' => [
        'data' => [],
        'pagination' => ['hasData' => false, 'totalPages' => 0, 'currentPage' => 0]
      ],
      'exporterData' => [
        'data' => [],
        'pagination' => ['hasData' => false, 'totalPages' => 0, 'currentPage' => 0]
      ]
    ];

    // Process CSV files if they exist
    if ($fileRecord) {
      // Load exporter data
      if (!empty($fileRecord['exporter_output_file'])) {
        $exporterResult = $this->loadTabData($fileRecord['exporter_output_file'], $page_number);
        $data['exporterData'] = $exporterResult;
      }

      // Load importer data  
      if (!empty($fileRecord['importer_output_file'])) {
        $importerResult = $this->loadTabData($fileRecord['importer_output_file'], $page_number);
        $data['importerData'] = $importerResult;
      }
    }
  
    return view('report', $data);
  }

  public function generateData()
  {
    $session = session();

    // Check if user is logged in
    if (!$session->get('logged_in')) {
      return $this->response->setJSON(['status' => 'error', 'message' => 'Unauthorized']);
    }

    // Get the single uploaded file record
    $fileRecord = $this->fileUploadModel->first();

    if (!$fileRecord) {
      return $this->response->setJSON(['status' => 'error', 'message' => 'No files found. Please upload files first.']);
    }

    // Process the files and generate output CSV files
    $result = $this->processAndGenerateReports($fileRecord);

    if ($result['success']) {
      return $this->response->setJSON(['status' => 'success', 'message' => 'Data generated successfully']);
    } else {
      return $this->response->setJSON(['status' => 'error', 'message' => $result['message']]);
    }
  }

  public function getTabData($type = 'importer', $page = 1)
  {
    $session = session();

    // Check if user is logged in
    if (!$session->get('logged_in')) {
      return $this->response->setJSON(['status' => 'error', 'message' => 'Unauthorized']);
    }

    try {
      $fileRecord = $this->fileUploadModel->first();

      if (!$fileRecord) {
        return $this->response->setJSON(['status' => 'error', 'message' => 'No data found']);
      }

      $csvFile = $type === 'importer' ? $fileRecord['importer_output_file'] : $fileRecord['exporter_output_file'];
      
      if (empty($csvFile)) {
        return $this->response->setJSON(['status' => 'error', 'message' => 'No ' . $type . ' file found']);
      }

      $result = $this->loadTabData($csvFile, $page - 1); // Convert to 0-based indexing

      return $this->response->setJSON($result);
      
    } catch (\Exception $e) {
      log_message('error', 'Error in getTabData: ' . $e->getMessage());
      return $this->response->setJSON(['status' => 'error', 'message' => 'Server error: ' . $e->getMessage()]);
    }
  }

  private function loadTabData($csvFileName, $pageNumber = 0, $perPage = 25)
  {
    $uploadPath = WRITEPATH . 'uploads/';
    
    if (empty($csvFileName) || !file_exists($uploadPath . $csvFileName)) {
      return [
        'data' => [],
        'pagination' => [
          'totalPages' => 0,
          'currentPage' => 0,
          'hasData' => false
        ]
      ];
    }

    $csvFilePath = $uploadPath . $csvFileName;
    $csvFile = fopen($csvFilePath, "r");
    $csvOutput = [];
    
    while (!feof($csvFile)) {
      $csvOutput[] = fgetcsv($csvFile);
    }
    fclose($csvFile);

    // Remove empty rows
    $csvOutput = array_filter($csvOutput, function($row) {
      return !empty($row) && !empty($row[0]);
    });

    // Separate headers and data
    $headers = array_shift($csvOutput); // Remove and get the first row (headers)
    $dataOnly = $csvOutput; // Remaining rows are data only

    // Calculate total entries
    $totalEntries = count($dataOnly);

    // Split data into chunks of 25 rows each
    $dataArray = array_chunk($dataOnly, $perPage);
    $totalPages = count($dataArray);
    
    // Get current page data (default to first page if page number is invalid)
    $currentPageData = isset($dataArray[$pageNumber]) ? $dataArray[$pageNumber] : [];

    // Always include headers in the response
    $responseData = [];
    if ($headers) {
      $responseData[] = $headers; // Add headers as first row
    }
    
    // Add the current page data
    foreach ($currentPageData as $row) {
      $responseData[] = $row;
    }

    return [
      'data' => $responseData,
      'pagination' => [
        'totalPages' => $totalPages,
        'currentPage' => $pageNumber,
        'hasData' => !empty($currentPageData),
        'perPage' => $perPage,
        'totalEntries' => $totalEntries
      ]
    ];
  }



  private function processAndGenerateReports($fileRecord)
  {
    try {
      $uploadPath = WRITEPATH . 'uploads/';
      
      // Check if required files exist
      if (empty($fileRecord['file_importer']) && empty($fileRecord['file_exporter'])) {
        return ['success' => false, 'message' => 'No importer or exporter files found'];
      }
      
      if (empty($fileRecord['source_file'])) {
        return ['success' => false, 'message' => 'Source file is required'];
      }

      $exporter_array_keys = $importer_array_keys = [];
      $source_data = [];

      // Process Source File
      if (!empty($fileRecord['source_file'])) {
        $sourceFilePath = $uploadPath . $fileRecord['source_file'];
        
        if (!file_exists($sourceFilePath)) {
          return ['success' => false, 'message' => 'Source file not found'];
        }

        $sourceFile = fopen($sourceFilePath, "r");
        $sourceDataSet = [];
        
        while (!feof($sourceFile)) {
          $sourceDataSet[] = fgetcsv($sourceFile);
        }
        fclose($sourceFile);

        // Remove header row and process data
        unset($sourceDataSet[0]);
        foreach ($sourceDataSet as $row) {
          if (!empty($row[0])) {
            $ice_id = $row[0];
            $source_data[$ice_id] = $row;
          }
        }
      }

      // Process Exporter File
      $exporter_data = [];
      if (!empty($fileRecord['file_exporter'])) {
        $exporterFilePath = $uploadPath . $fileRecord['file_exporter'];
        
        if (file_exists($exporterFilePath)) {
          $exporterFile = fopen($exporterFilePath, "r");
          $exporterFileDataSet = [];
          
          while (!feof($exporterFile)) {
            $exporterFileDataSet[] = fgetcsv($exporterFile);
          }
          fclose($exporterFile);

          // Remove header row and process data
          unset($exporterFileDataSet[0]);
          foreach ($exporterFileDataSet as $row) {
            if (!empty($row[0])) {
              $ex_ice_id = isset($row[1]) ? trim($row[1]) : '';
              if ($ex_ice_id) {
                $exporter_data[$ex_ice_id][] = $row;
              }
            }
          }
          $exporter_array_keys = array_filter(array_keys($exporter_data));
        }
      }

      // Process Importer File
      $importer_data = [];
      if (!empty($fileRecord['file_importer'])) {
        $importerFilePath = $uploadPath . $fileRecord['file_importer'];
        
        if (file_exists($importerFilePath)) {
          $importerFile = fopen($importerFilePath, "r");
          $importerFileDataSet = [];
          
          while (!feof($importerFile)) {
            $importerFileDataSet[] = fgetcsv($importerFile);
          }
          fclose($importerFile);

          // Remove header row and process data
          unset($importerFileDataSet[0]);
          foreach ($importerFileDataSet as $row) {
            if (!empty($row[0])) {
              $im_ice_id = isset($row[1]) ? trim($row[1]) : '';
              if ($im_ice_id) {
                $importer_data[$im_ice_id][] = $row;
              }
            }
          }
          $importer_array_keys = array_filter(array_keys($importer_data));
        }
      }

      $output_importer_data = $output_exporter_data = [];

      // Process Importer Data
      if (count($importer_array_keys) > 0) {
        foreach ($importer_array_keys as $iak => $iav) {
          if ($iav) {
            if (isset($source_data[$iav]) && !empty($source_data[$iav])) {
              $importer_source_data_value = $source_data[$iav];
              $importer_name = $importer_source_data_value[1];
              $ms_name = $importer_source_data_value[9];
              $ice_no = $importer_source_data_value[0];
              $address = $importer_source_data_value[2] . $importer_source_data_value[3] . ',' . $importer_source_data_value[4] . ',' . $importer_source_data_value[6] . '-' . $importer_source_data_value[5];
              $city = $importer_source_data_value[4];
              $contact_no = $importer_source_data_value[7];
              $email = $importer_source_data_value[8];
              
              if (isset($importer_data[$iav]) && !empty($importer_data[$iav])) {
                $bcd_value_array = array_column($importer_data[$iav], 19);
                $bcd_value = 0;
                foreach ($bcd_value_array as $scV) {
                  $bcd_value += floatval($scV);
                }
              } else {
                $bcd_value = 0;
              }
              
              if (isset($exporter_data[$iav]) && !empty($exporter_data[$iav])) {
                $number_of_scroll = count($exporter_data[$iav]);
                $scroll_amount_array = array_column($exporter_data[$iav], 7);
                $scroll_amount = 0;
                foreach ($scroll_amount_array as $scV) {
                  $amount = str_replace(',', '', $scV);
                  $scroll_amount += floatval($amount);
                }
              } else {
                $number_of_scroll = $scroll_amount = 0;
              }
              
              $total_amount = ($bcd_value > 0) ? ($bcd_value - $scroll_amount) : 0;
              $com_ms = 'Kindly Att: ' . $ms_name;
              $com_address = $address;
              
              $output_importer_data[] = [++$iak, $ice_no, $importer_name, $com_ms, $com_address, $city, $bcd_value, $number_of_scroll, $scroll_amount, $total_amount, $contact_no, $email, '-'];
            }
          }
        }
      }

      // Process Exporter Data
      if (count($exporter_array_keys) > 0) {
        foreach ($exporter_array_keys as $eak => $eav) {
          if ($eav) {
            if (isset($source_data[$eav]) && !empty($source_data[$eav])) {
              $exporter_source_data_value = $source_data[$eav];
              $ice_no = $exporter_source_data_value[0];
              $ms_name = $exporter_source_data_value[9];
              $city = $exporter_source_data_value[4];
              $contact_no = $exporter_source_data_value[7];
              $email = $exporter_source_data_value[8];
              $address = $exporter_source_data_value[2] . $exporter_source_data_value[3] . ',' . $exporter_source_data_value[4] . ',' . $exporter_source_data_value[6] . '-' . $exporter_source_data_value[5];
              $exporter_name = $exporter_data[$eav][0][2];
              $number_of_scroll = count($exporter_data[$eav]);

              $scroll_amount_array = array_column($exporter_data[$eav], 7);
              $scroll_amount = 0;
              foreach ($scroll_amount_array as $scV) {
                $amount = str_replace(',', '', $scV);
                $scroll_amount += floatval($amount);
              }

              if (isset($importer_data[$eav]) && !empty($importer_data[$eav])) {
                $bcd_value_array = array_column($importer_data[$eav], 19);
                $bcd_value = 0;
                foreach ($bcd_value_array as $scV) {
                  $bcd_value += floatval($scV);
                }
              } else {
                $bcd_value = 0;
              }

              $total_amount = ($bcd_value > 0) ? ($bcd_value - $scroll_amount) : ($scroll_amount - $bcd_value);
              $com_ms = 'Kindly Att: ' . $ms_name;
              $com_address = $address;
              
              $output_exporter_data[] = [++$eak, $ice_no, $exporter_name, $com_ms, $com_address, $city, $contact_no, $email, $number_of_scroll, $scroll_amount, $bcd_value, $total_amount, '-'];
            }
          }
        }
      }

      // Generate Output Files
      $uploadPath = WRITEPATH . 'uploads/';

      // Generate Exporter Output File
      if (count($output_exporter_data) > 0) {
        $ex_data_file_name = mt_rand() . time() . '_output_exporter_data.csv';
        $ex_file1 = fopen($uploadPath . $ex_data_file_name, "w");
        fputcsv($ex_file1, ['Serial Number', 'IEC', 'Exporter Name', 'M/s.', 'Address', 'City', 'Contact Number', 'Email id', 'Number of Scroll', 'Scroll Value', 'BCD value', '(BCD - Scroll value)', 'Remarks']);
        
        foreach ($output_exporter_data as $line) {
          if (isset($line[0]) && !empty($line[0])) {
            fputcsv($ex_file1, $line);
          }
        }
        fclose($ex_file1);
        
        // Delete old exporter output file if exists
        if (!empty($fileRecord['exporter_output_file'])) {
          @unlink($uploadPath . $fileRecord['exporter_output_file']);
        }
      } else {
        $ex_data_file_name = null;
        // Delete old exporter output file if exists
        if (!empty($fileRecord['exporter_output_file'])) {
          @unlink($uploadPath . $fileRecord['exporter_output_file']);
        }
      }

      // Generate Importer Output File
      if (count($output_importer_data) > 0) {
        $im_data_file_name = mt_rand() . time() . '_output_importer_data.csv';
        $im_file1 = fopen($uploadPath . $im_data_file_name, "w");
        fputcsv($im_file1, ['Serial Number', 'IEC', 'Importer Name', 'M/s.', 'Address', 'City', 'BCD value', 'Number of Scroll ', 'Scroll Value', '(BCD - Scrolls value)', 'Contact Number', 'Email id ', 'Remarks']);
        
        foreach ($output_importer_data as $line) {
          if (isset($line[0]) && !empty($line[0])) {
            fputcsv($im_file1, $line);
          }
        }
        fclose($im_file1);
        
        // Delete old importer output file if exists
        if (!empty($fileRecord['importer_output_file'])) {
          @unlink($uploadPath . $fileRecord['importer_output_file']);
        }
      } else {
        $im_data_file_name = null;
        // Delete old importer output file if exists
        if (!empty($fileRecord['importer_output_file'])) {
          @unlink($uploadPath . $fileRecord['importer_output_file']);
        }
      }

      // Update database with new output files
      $updateData = [
        'importer_output_file' => $im_data_file_name,
        'exporter_output_file' => $ex_data_file_name,
        'media_modification_time' => date('Y-m-d H:i:s')
      ];

      $this->fileUploadModel->update($fileRecord['id'], $updateData);

      return ['success' => true];
      
    } catch (\Exception $e) {
      return ['success' => false, 'message' => $e->getMessage()];
    }
  }
}